
(function () {
  const api = window.wp?.apiFetch;
  if (!api) return;

  api.use((options, next) => {
    options.headers = options.headers || {};
    if (window.FSTRB?.nonce) options.headers['X-WP-Nonce'] = window.FSTRB.nonce;
    return next(options);
  });

  function qs(el, sel) { return el.querySelector(sel); }
  function qsa(el, sel) { return Array.from(el.querySelectorAll(sel)); }
  function fmtMonthTitle(ym) {
    const [y, m] = ym.split('-').map(x => parseInt(x, 10));
    const d = new Date(y, m - 1, 1);
    return d.toLocaleString(window.FSTRB?.locale || undefined, { month: 'long', year: 'numeric' });
  }

  async function loadServices(box, unitId) {
    try {
      const locale = window.FSTRB?.locale || '';
      const currency = box.closest('.fstrb-booking')?.dataset.currency || 'EUR';
      const res = await api({ path: `/fstrb/v1/services?unit_id=${unitId}&locale=${locale}&currency=${currency}` });
      const services = res.services || [];
      if (!services.length) {
        box.innerHTML = `<div style="opacity:.7">${window.FSTRB?.i18n?.noServices || 'No optional services.'}</div>`;
        return;
      }
      box.innerHTML = services.map(s => {
        const typeLabel = window.FSTRB?.i18n?.[s.pricing_type] || s.pricing_type;

        let metaHtml = '';
        if (s.meta_data && s.meta_data.length) {
          const rows = s.meta_data.map(m => `
                <div class="fstrb-svc-meta fstrb-meta-row" style="font-size:0.9em;color:#666">
                    <strong class="fstrb-meta-key">${escapeHtml(m.key)}:</strong> ${escapeHtml(m.value)}
                </div>
             `).join('');
          metaHtml = `<div class="fstrb-meta-container" style="margin-left:24px;margin-top:2px">${rows}</div>`;
        }

        return `
        <div class="fstrb-svc">
          <label style="display:flex;gap:10px;align-items:center">
            <input type="checkbox" class="fstrb-svc-check" value="${s.id}" ${s.is_optional == 0 ? 'checked disabled' : ''}>
            <span>${escapeHtml(s.name)} <small>(${typeLabel}, ${Number(s.price).toFixed(2)} ${escapeHtml(currency)})</small></span>
          </label>
          ${metaHtml}
        </div>
      `;
      }).join('');
    } catch (e) {
      box.innerHTML = `<div style="color:#b00">${window.FSTRB?.i18n?.errorLoadingServices || 'Error loading services.'}</div>`;
    }
  }

  async function loadContactFields(box, unitId) {
    try {
      const locale = window.FSTRB?.locale || '';
      const res = await api({ path: `/fstrb/v1/contact-fields?unit_id=${unitId}&locale=${locale}` });
      const fields = res.fields || [];
      if (!fields.length) return;

      box.innerHTML = fields.map(f => `
        <div class="fstrb-row fstrb-custom-field-row" style="margin-bottom:10px">
          <label>${escapeHtml(f.label)}</label>
          <input type="text" class="fstrb-custom-input" data-key="${escapeHtml(f.key)}" style="width:100%" />
        </div>
      `).join('');
    } catch (e) {
      console.error('Error loading contact fields', e);
    }
  }

  function escapeHtml(s) {
    return String(s).replace(/[&<>"']/g, (c) => ({ "&": "&amp;", "<": "&lt;", ">": "&gt;", '"': "&quot;", "'": "&#039;" }[c]));
  }

  async function quote(root) {
    const unitId = root.dataset.unit;
    const df = qs(root, '.fstrb-date-from')?.value;
    const dt = qs(root, '.fstrb-date-to')?.value;
    const currency = root.dataset.currency || 'EUR';
    const guests = parseInt(qs(root, '.fstrb-guests')?.value || '1', 10);
    const checks = qsa(root, '.fstrb-svc-check:checked').map(i => parseInt(i.value, 10));
    if (!df || !dt) return;

    const out = qs(root, '.fstrb-total');
    try {
      const res = await api({
        path: '/fstrb/v1/quote',
        method: 'POST',
        data: { unit_id: parseInt(unitId, 10), date_from: df, date_to: dt, guests, services: checks, currency }
      });
      out.textContent = (res.total ?? 0).toFixed(2);
      if (res.discount_total > 0) {
        const discLabel = window.FSTRB?.i18n?.discount || 'Discount';
        out.innerHTML += ` <span class="fstrb-discount" style="color:#27ae60;font-size:0.9em;margin-left:5px">(${discLabel} -${Number(res.discount_total).toFixed(2)})</span>`;
      }
    } catch (e) {
      if (e?.data?.error === 'min_stay_violation') out.textContent = (window.FSTRB?.i18n?.minStay || 'Minimum stay is %d nights.').replace('%d', e.data.limit);
      else if (e?.data?.error === 'max_stay_violation') out.textContent = (window.FSTRB?.i18n?.maxStay || 'Maximum stay is %d nights.').replace('%d', e.data.limit);
      else out.textContent = '';
    }
  }

  async function book(root) {
    const unitId = parseInt(root.dataset.unit, 10);
    const df = qs(root, '.fstrb-date-from')?.value;
    const dt = qs(root, '.fstrb-date-to')?.value;
    const currency = root.dataset.currency || 'EUR';
    const guests = parseInt(qs(root, '.fstrb-guests')?.value || '1', 10);
    const services = qsa(root, '.fstrb-svc-check:checked').map(i => parseInt(i.value, 10));
    const name = qs(root, '.fstrb-name')?.value || '';
    const lastname = qs(root, '.fstrb-lastname')?.value || '';
    const gender = qs(root, '.fstrb-gender')?.value || 'male';
    const email = qs(root, '.fstrb-email')?.value || '';
    const phonePrefix = qs(root, '.fstrb-phone-prefix')?.value || '';
    const phoneNum = qs(root, '.fstrb-phone')?.value || '';
    const phone = phonePrefix + phoneNum;
    const note = qs(root, '.fstrb-note')?.value || '';
    const hp = qs(root, '.fstrb-honeypot')?.value || '';
    const captcha = qs(root, '.fstrb-captcha')?.value || '';

    const customFields = {};
    qsa(root, '.fstrb-custom-input').forEach(input => {
      customFields[input.dataset.key] = input.value;
    });

    const msg = qs(root, '.fstrb-msg');
    const confirmBox = qs(root, '.fstrb-confirm-now');
    const confirmNow = confirmBox && confirmBox.checked ? 1 : 0;

    msg.textContent = window.FSTRB?.i18n?.loading || 'Sending…';

    const restUrl = window.FSTRB?.restUrl || '/wp-json/fstrb/v1';
    const nonce = window.FSTRB?.nonce || '';

    if (!df || !dt) {
      msg.textContent = window.FSTRB?.i18n?.selectDates || 'Select dates.';
      return;
    }

    if (!name || !lastname || !email || !phoneNum || !captcha) {
      msg.textContent = window.FSTRB?.i18n?.missingContact || 'Fill in all required fields.';
      return;
    }

    try {
      const res = await api({
        path: '/fstrb/v1/book',
        method: 'POST',
        data: { unit_id: unitId, date_from: df, date_to: dt, guests, services, name, lastname, gender, email, phone, note, confirm_now: confirmNow, hp, captcha, custom_fields: customFields, currency }
      });

      // Success: Hide form and show thanks message
      root.style.display = 'none';
      const thanks = root.parentNode.querySelector('.fstrb-thanks');
      if (thanks) {
        thanks.style.display = 'block';
        thanks.scrollIntoView({ behavior: 'smooth', block: 'center' });
      } else {
        msg.textContent = `${window.FSTRB?.i18n?.success || 'Done.'} #${res.booking_id} (${res.status}).`;
      }
    } catch (e) {
      if (e?.data?.error === 'date_conflict') msg.textContent = window.FSTRB?.i18n?.occupied || 'Dates are already occupied.';
      else if (e?.data?.error === 'missing_contact') msg.textContent = window.FSTRB?.i18n?.missingContact || 'Provide name and email.';
      else if (e?.data?.error === 'min_stay_violation') msg.textContent = (window.FSTRB?.i18n?.minStay || 'Minimum stay is %d nights.').replace('%d', e.data.limit);
      else if (e?.data?.error === 'max_stay_violation') msg.textContent = (window.FSTRB?.i18n?.maxStay || 'Maximum stay is %d nights.').replace('%d', e.data.limit);
      else if (e?.data?.error === 'captcha_invalid') msg.textContent = window.FSTRB?.i18n?.captchaError || 'Captcha failed.';
      else if (e?.data?.error === 'robot_detected') msg.textContent = window.FSTRB?.i18n?.robotError || 'Robot detected.';
      else msg.textContent = window.FSTRB?.i18n?.error || 'Error sending.';

      // Append detailed error if available and not one of the above known ones for better UX/Debugging
      if (e?.data?.error && !['date_conflict', 'missing_contact', 'min_stay_violation', 'max_stay_violation', 'captcha_invalid', 'robot_detected'].includes(e.data.error)) {
        msg.textContent += ' (' + e.data.error + ')';
      }
    }
  }

  function initBooking() {
    document.querySelectorAll('.fstrb-booking').forEach(async (root) => {
      // Admin: Inject Confirm Checkbox
      if (window.FSTRB?.isAdmin) {
        const btnRow = qs(root, '.fstrb-submit')?.parentNode;
        if (btnRow) {
          const div = document.createElement('div');
          div.className = 'fstrb-row';
          div.style.marginBottom = '10px';
          div.innerHTML = `<label style="background:#ffc;padding:5px;display:inline-block;border:1px solid #ee9"><input type="checkbox" class="fstrb-confirm-now" value="1"> ${window.FSTRB?.i18n?.confirmNow || 'Confirm immediately (Admin)'}</label>`;
          btnRow.parentNode.insertBefore(div, btnRow);
        }
      }

      const unitId = parseInt(root.dataset.unit, 10);
      const box = qs(root, '.fstrb-services-box');
      if (box) await loadServices(box, unitId);

      const customBox = qs(root, '.fstrb-custom-fields');
      if (customBox) await loadContactFields(customBox, unitId);

      ['change', 'input'].forEach(ev => {
        root.addEventListener(ev, (e) => {
          if (e.target.matches('.fstrb-date-from,.fstrb-date-to,.fstrb-guests,.fstrb-svc-check')) quote(root);
        });
      });

      const btn = qs(root, '.fstrb-submit');
      if (btn) btn.addEventListener('click', (e) => { e.preventDefault(); book(root); });
    });
  }

  function buildMonthGrid(container, month, occupiedSet, ranges = []) {
    const [y, m] = month.split('-').map(n => parseInt(n, 10));
    const first = new Date(y, m - 1, 1);
    const startDow = (first.getDay() + 6) % 7; // 0=Mon
    const daysInMonth = new Date(y, m, 0).getDate();

    const today = new Date();
    today.setHours(0, 0, 0, 0);

    const weekdays = [
      window.FSTRB?.i18n?.mon || 'Mo',
      window.FSTRB?.i18n?.tue || 'Tu',
      window.FSTRB?.i18n?.wed || 'We',
      window.FSTRB?.i18n?.thu || 'Th',
      window.FSTRB?.i18n?.fri || 'Fr',
      window.FSTRB?.i18n?.sat || 'Sa',
      window.FSTRB?.i18n?.sun || 'Su'
    ];

    const monthBlock = document.createElement('div');
    monthBlock.className = 'fstrb-month-block';
    monthBlock.innerHTML = `
      <div class="fstrb-month-title">${fmtMonthTitle(month)}</div>
      <div class="fstrb-cal-grid"></div>
    `;
    const grid = monthBlock.querySelector('.fstrb-cal-grid');

    const cells = [];
    weekdays.forEach(w => cells.push(`<div class="fstrb-cal-wd">${w}</div>`));
    for (let i = 0; i < startDow; i++) cells.push(`<div></div>`);

    for (let d = 1; d <= daysInMonth; d++) {
      const day = `${month}-${String(d).padStart(2, '0')}`;
      const occ = occupiedSet.has(day);
      const isCheckin = ranges.some(r => r.date_from === day);
      const isCheckout = ranges.some(r => r.date_to === day);

      const parts = day.split('-').map(Number);
      const dObj = new Date(parts[0], parts[1] - 1, parts[2]);
      const isPast = dObj < today;

      let cls = 'fstrb-cal-day';
      if (occ) cls += ' occ';
      if (isPast) cls += ' past';

      if (isCheckin && isCheckout) cls += ' fstrb-checkout-checkin';
      else if (isCheckin) cls += ' fstrb-checkin';
      else if (isCheckout) cls += ' fstrb-checkout';

      if (!occ && !isPast) cls += ' free';

      let tag = occ ? (window.FSTRB?.i18n?.occupied || 'occupied') : (isPast ? '' : (window.FSTRB?.i18n?.free || 'available'));
      if (isCheckout && !occ && !isPast) tag = window.FSTRB?.i18n?.checkout || 'checkout';
      if (isCheckin && isCheckout) tag = window.FSTRB?.i18n?.transition || 'transition';

      cells.push(`
        <div class="${cls}" data-day="${day}">
          <div class="d">${d}</div>
          <div class="tag">${tag}</div>
        </div>
      `);
    }
    grid.innerHTML = cells.join('');
    container.appendChild(monthBlock);
  }

  function initCalendar() {
    document.querySelectorAll('.fstrb-calendar').forEach(async (root) => {
      const unitId = parseInt(root.dataset.unit, 10);
      const monthsCount = parseInt(root.dataset.months || '1', 10);
      const gridCols = parseInt(root.dataset.grid || '1', 10);
      const gridContainer = qs(root, '.fstrb-cal-grid-container');

      if (gridContainer) {
        gridContainer.style.setProperty('--fstrb-grid', gridCols);
      }

      if (monthsCount === 1) {
        root.classList.add('fstrb-single-month');
      } else {
        root.classList.remove('fstrb-single-month');
      }

      const selectedLabel = qs(root, '.fstrb-selected-range');
      const bookingRoot = root.closest('.fstrb-booking');

      let cur = new Date();
      let selFrom = null;
      let selTo = null;

      async function load() {
        const startMonth = `${cur.getFullYear()}-${String(cur.getMonth() + 1).padStart(2, '0')}`;

        const res = await api({ path: `/fstrb/v1/calendar?unit_id=${unitId}&month=${startMonth}&months=${monthsCount}` });
        const occupied = new Set(res.occupiedDays || []);
        const ranges = res.ranges || [];

        gridContainer.innerHTML = '';
        for (let i = 0; i < monthsCount; i++) {
          const mDate = new Date(cur.getFullYear(), cur.getMonth() + i, 1);
          const monthStr = `${mDate.getFullYear()}-${String(mDate.getMonth() + 1).padStart(2, '0')}`;
          buildMonthGrid(gridContainer, monthStr, occupied, ranges);
        }

        // Hide main header title if multi-month
        const mainTitle = qs(root, '.fstrb-cal-title');
        if (monthsCount > 1) {
          mainTitle.style.display = 'none';
        } else {
          mainTitle.textContent = fmtMonthTitle(startMonth);
          mainTitle.style.display = '';
        }

        function clearSel() {
          qsa(gridContainer, '.fstrb-cal-day').forEach(el => el.classList.remove('sel', 'inrange'));
        }
        function paintSel() {
          clearSel();
          if (!selFrom) return;
          const from = selFrom;
          const to = selTo;
          qsa(gridContainer, '.fstrb-cal-day').forEach(el => {
            const day = el.dataset.day;
            if (day === from) el.classList.add('sel');
            if (to && day === to) el.classList.add('sel');
            if (to && day > from && day < to) el.classList.add('inrange');
          });
          selectedLabel.textContent = to ? `${from} → ${to}` : `${from} → …`;

          if (bookingRoot) {
            const inputFrom = qs(bookingRoot, '.fstrb-date-from');
            const inputTo = qs(bookingRoot, '.fstrb-date-to');
            if (inputFrom) inputFrom.value = from || '';
            if (inputTo) inputTo.value = to || '';
            if (from && to) quote(bookingRoot);
          }
        }

        const legend = qs(root, '.fstrb-cal-legend');
        const selected = qs(root, '.fstrb-cal-selected');
        if (!bookingRoot) {
          if (legend) legend.style.display = 'none';
          if (selected) selected.style.display = 'none';
        }

        gridContainer.onclick = (e) => {
          if (!bookingRoot) return;
          const cell = e.target.closest('.fstrb-cal-day');
          if (!cell || cell.classList.contains('past')) return;

          const isOcc = cell.classList.contains('occ');
          const isCheckin = cell.classList.contains('fstrb-checkin') || cell.classList.contains('fstrb-checkout-checkin');
          const isCheckout = cell.classList.contains('fstrb-checkout') || cell.classList.contains('fstrb-checkout-checkin');
          const day = cell.dataset.day;

          // If selecting start date (or reset)
          if (!selFrom || (selFrom && selTo)) {
            if (isOcc && !isCheckout) return;
            selFrom = day; selTo = null; paintSel(); return;
          }

          // Selecting end date: must be after start
          if (day <= selFrom) {
            if (isOcc && !isCheckout) return;
            selFrom = day; selTo = null; paintSel(); return;
          }

          // Rule for end date: can be free or a check-in day for someone else
          if (isOcc && !isCheckin) return;

          // check all days between are free and not past
          const days = [];
          const dParts = selFrom.split('-').map(Number);
          let d = new Date(dParts[0], dParts[1] - 1, dParts[2]);
          const eParts = day.split('-').map(Number);
          const end = new Date(eParts[0], eParts[1] - 1, eParts[2]);

          while (d < end) {
            const s = `${d.getFullYear()}-${String(d.getMonth() + 1).padStart(2, '0')}-${String(d.getDate()).padStart(2, '0')}`;
            days.push(s);
            d.setDate(d.getDate() + 1);
          }
          const conflict = days.some(x => occupied.has(x));
          if (conflict) {
            if (window.FSTRB?.i18n?.occupied) alert(window.FSTRB.i18n.occupied);
            selFrom = null; selTo = null; clearSel(); selectedLabel.textContent = ''; return;
          }
          selTo = day; paintSel();
        };

        paintSel();
      }

      qs(root, '.fstrb-prev')?.addEventListener('click', () => { cur.setMonth(cur.getMonth() - 1); load(); });
      qs(root, '.fstrb-next')?.addEventListener('click', () => { cur.setMonth(cur.getMonth() + 1); load(); });

      await load();
    });
  }

  document.addEventListener('DOMContentLoaded', () => {
    initBooking();
    initCalendar();
  });
})();
