<?php
if (!defined('ABSPATH')) exit;

class FSTRB_Admin {
	public static function register() {
		add_menu_page(
			__('Bookings', 'free-short-term-rental-booking'),
			__('Bookings', 'free-short-term-rental-booking'),
			'manage_options',
			'fstrb',
			[__CLASS__, 'page_dashboard'],
			'dashicons-calendar-alt',
			26
		);

		add_submenu_page('fstrb', __('Calendar', 'free-short-term-rental-booking'), __('Calendar', 'free-short-term-rental-booking'), 'manage_options', 'fstrb-calendar', [__CLASS__, 'page_calendar']);
		add_submenu_page('fstrb', __('General', 'free-short-term-rental-booking'), __('General', 'free-short-term-rental-booking'), 'manage_options', 'fstrb-general', [__CLASS__, 'page_general']);
		add_submenu_page('fstrb', __('Rates', 'free-short-term-rental-booking'), __('Rates', 'free-short-term-rental-booking'), 'manage_options', 'fstrb-rates', [__CLASS__, 'page_rates']);
		add_submenu_page('fstrb', __('Services', 'free-short-term-rental-booking'), __('Services', 'free-short-term-rental-booking'), 'manage_options', 'fstrb-services', [__CLASS__, 'page_services']);
		add_submenu_page('fstrb', __('Styling', 'free-short-term-rental-booking'), __('Styling', 'free-short-term-rental-booking'), 'manage_options', 'fstrb-styling', [__CLASS__, 'page_styling']);
		add_submenu_page('fstrb', __('iCal Imports', 'free-short-term-rental-booking'), __('iCal Imports', 'free-short-term-rental-booking'), 'manage_options', 'fstrb-ical', [__CLASS__, 'page_ical']);
		add_submenu_page('fstrb', __('Contact Fields', 'free-short-term-rental-booking'), __('Contact Fields', 'free-short-term-rental-booking'), 'manage_options', 'fstrb-contacts', [__CLASS__, 'page_contact_fields']);
	}

	public static function page_dashboard() {
		global $wpdb;
		$t = FSTRB_DB::tables();

		// Handle Actions
		if (isset($_GET['action']) && isset($_GET['id']) && isset($_GET['_wpnonce'])) {
			$id = (int)$_GET['id'];
			if (wp_verify_nonce($_GET['_wpnonce'], 'fstrb_booking_action_' . $id)) {
				if ($_GET['action'] === 'confirm') {
					$wpdb->update($t['bookings'], ['status'=>'confirmed'], ['id'=>$id]);
					FSTRB_Emails::notify_customer_confirmed($id);
					echo '<div class="notice notice-success"><p>' . __('Booking confirmed.', 'free-short-term-rental-booking') . '</p></div>';
				}
				if ($_GET['action'] === 'delete') {
					// Delete dependencies
					$wpdb->delete($t['booking_days'], ['booking_id'=>$id]);
					$wpdb->delete($t['booking_services'], ['booking_id'=>$id]);
					$wpdb->delete($t['bookings'], ['id'=>$id]);
					echo '<div class="notice notice-success"><p>' . __('Booking deleted.', 'free-short-term-rental-booking') . '</p></div>';
				}
			} else {
				 echo '<div class="notice notice-error"><p>' . __('Nonce verification failed.', 'free-short-term-rental-booking') . '</p></div>';
			}
		}

		echo '<div class="wrap"><h1>' . __('Bookings', 'free-short-term-rental-booking') . '</h1>';

		// Get filter parameters
		$filter_unit = isset($_GET['filter_unit']) ? (int)$_GET['filter_unit'] : 0;
		$filter_status = isset($_GET['filter_status']) ? sanitize_text_field($_GET['filter_status']) : '';
		$filter_source = isset($_GET['filter_source']) ? sanitize_text_field($_GET['filter_source']) : '';
		$filter_date_from = isset($_GET['filter_date_from']) ? sanitize_text_field($_GET['filter_date_from']) : '';
		$filter_date_to = isset($_GET['filter_date_to']) ? sanitize_text_field($_GET['filter_date_to']) : '';
		$filter_guest = isset($_GET['filter_guest']) ? sanitize_text_field($_GET['filter_guest']) : '';
		$filter_booking_id = isset($_GET['filter_booking_id']) ? sanitize_text_field($_GET['filter_booking_id']) : '';

		// Get sorting parameters
		$orderby = isset($_GET['orderby']) ? sanitize_text_field($_GET['orderby']) : 'id';
		$order = isset($_GET['order']) && strtoupper($_GET['order']) === 'ASC' ? 'ASC' : 'DESC';

		// Get pagination parameters
		$per_page = isset($_GET['per_page']) ? (int)$_GET['per_page'] : 50;
		$paged = isset($_GET['paged']) ? max(1, (int)$_GET['paged']) : 1;
		$offset = ($paged - 1) * $per_page;

		// Display filter form
		echo '<form method="get" class="fstrb-filters" style="background:#fff;padding:15px;margin:15px 0;border:1px solid #ccd0d4">';
		echo '<input type="hidden" name="page" value="fstrb" />';
		echo '<h3 style="margin-top:0">' . __('Filters', 'free-short-term-rental-booking') . '</h3>';
		echo '<div style="display:grid;grid-template-columns:repeat(auto-fit,minmax(200px,1fr));gap:10px;margin-bottom:10px">';
		
		// Unit filter
		echo '<div><label>' . __('Unit', 'free-short-term-rental-booking') . '<br>';
		$units = get_posts(['post_type'=>'fstrb_unit','numberposts'=>-1,'post_status'=>['publish','draft']]);
		echo '<select name="filter_unit" style="width:100%">';
		echo '<option value="0">' . __('All Units', 'free-short-term-rental-booking') . '</option>';
		foreach ($units as $u) {
			printf('<option value="%d" %s>%s</option>', $u->ID, selected($filter_unit,$u->ID,false), esc_html($u->post_title));
		}
		echo '</select></label></div>';

		// Status filter
		echo '<div><label>' . __('Status', 'free-short-term-rental-booking') . '<br>';
		echo '<select name="filter_status" style="width:100%">';
		echo '<option value="">' . __('All Statuses', 'free-short-term-rental-booking') . '</option>';
		$statuses = ['pending'=>__('Pending', 'free-short-term-rental-booking'), 'confirmed'=>__('Confirmed', 'free-short-term-rental-booking'), 'cancelled'=>__('Cancelled', 'free-short-term-rental-booking'), 'blocked'=>__('Blocked', 'free-short-term-rental-booking')];
		foreach ($statuses as $k=>$v) {
			printf('<option value="%s" %s>%s</option>', $k, selected($filter_status,$k,false), $v);
		}
		echo '</select></label></div>';

		// Source filter
		echo '<div><label>' . __('Source', 'free-short-term-rental-booking') . '<br>';
		echo '<select name="filter_source" style="width:100%">';
		echo '<option value="">' . __('All Sources', 'free-short-term-rental-booking') . '</option>';
		$sources = [
			'direct' => __('Direct', 'free-short-term-rental-booking'),
			'admin' => __('Admin', 'free-short-term-rental-booking'),
			'ical_airbnb' => 'iCal Airbnb',
			'ical_booking' => 'iCal Booking.com',
			'ical_e_chalupy' => 'iCal e-chalupy.cz',
			'ical_megaubytovanie' => 'iCal Megaubytovanie',
			'ical_hauzi' => 'iCal Hauzi',
			'ical_bezrealitky' => 'iCal BezRealitky',
		];
		foreach ($sources as $k=>$v) {
			printf('<option value="%s" %s>%s</option>', $k, selected($filter_source,$k,false), $v);
		}
		echo '</select></label></div>';

		// Date from filter
		echo '<div><label>' . __('Date From', 'free-short-term-rental-booking') . '<br>';
		echo '<input type="date" name="filter_date_from" value="'.esc_attr($filter_date_from).'" style="width:100%"></label></div>';

		// Date to filter
		echo '<div><label>' . __('Date To', 'free-short-term-rental-booking') . '<br>';
		echo '<input type="date" name="filter_date_to" value="'.esc_attr($filter_date_to).'" style="width:100%"></label></div>';

		// Guest name filter
		echo '<div><label>' . __('Guest Name', 'free-short-term-rental-booking') . '<br>';
		echo '<input type="text" name="filter_guest" value="'.esc_attr($filter_guest).'" placeholder="'.__('Search...', 'free-short-term-rental-booking').'" style="width:100%"></label></div>';

		// Booking ID filter
		echo '<div><label>' . __('Booking/Reservation ID', 'free-short-term-rental-booking') . '<br>';
		echo '<input type="text" name="filter_booking_id" value="'.esc_attr($filter_booking_id).'" placeholder="'.__('ID or Reservation #', 'free-short-term-rental-booking').'" style="width:100%"></label></div>';

		echo '</div>';
		echo '<button type="submit" class="button button-primary">' . __('Apply Filters', 'free-short-term-rental-booking') . '</button> ';
		echo '<a href="'.admin_url('admin.php?page=fstrb').'" class="button">' . __('Clear Filters', 'free-short-term-rental-booking') . '</a>';
		echo '</form>';

		// Build WHERE clause
		$where = ['1=1'];
		$where_values = [];

		if ($filter_unit > 0) {
			$where[] = 'b.unit_id = %d';
			$where_values[] = $filter_unit;
		}
		if ($filter_status) {
			$where[] = 'b.status = %s';
			$where_values[] = $filter_status;
		}
		if ($filter_source) {
			$where[] = 'b.source = %s';
			$where_values[] = $filter_source;
		}
		if ($filter_date_from) {
			$where[] = 'b.date_from >= %s';
			$where_values[] = $filter_date_from;
		}
		if ($filter_date_to) {
			$where[] = 'b.date_to <= %s';
			$where_values[] = $filter_date_to;
		}
		if ($filter_guest) {
			$where[] = '(b.customer_name LIKE %s OR b.customer_lastname LIKE %s OR b.customer_email LIKE %s)';
			$like_guest = '%' . $wpdb->esc_like($filter_guest) . '%';
			$where_values[] = $like_guest;
			$where_values[] = $like_guest;
			$where_values[] = $like_guest;
		}
		if ($filter_booking_id) {
			$where[] = '(b.id = %d OR b.external_booking_id LIKE %s OR b.ical_uid LIKE %s)';
			$where_values[] = (int)$filter_booking_id;
			$like_id = '%' . $wpdb->esc_like($filter_booking_id) . '%';
			$where_values[] = $like_id;
			$where_values[] = $like_id;
		}

		$where_clause = implode(' AND ', $where);

		// Build ORDER BY clause
		$allowed_orderby = ['id', 'status', 'source', 'unit_title', 'date_from', 'date_to', 'customer_name', 'created_at'];
		if (!in_array($orderby, $allowed_orderby)) $orderby = 'id';
		$order_clause = "b.$orderby $order";
		if ($orderby === 'unit_title') $order_clause = "p.post_title $order";

		// Get total count
		$count_sql = "SELECT COUNT(*) FROM {$t['bookings']} b 
					 LEFT JOIN {$wpdb->posts} p ON p.ID = b.unit_id 
					 WHERE $where_clause";
		if ($where_values) {
			$count_sql = $wpdb->prepare($count_sql, $where_values);
		}
		$total_items = (int)$wpdb->get_var($count_sql);
		$total_pages = ceil($total_items / $per_page);

		// Get bookings
		$sql = "SELECT b.*, p.post_title as unit_title 
				FROM {$t['bookings']} b 
				LEFT JOIN {$wpdb->posts} p ON p.ID = b.unit_id 
				WHERE $where_clause
				ORDER BY $order_clause
				LIMIT %d OFFSET %d";
		
		$query_values = array_merge($where_values, [$per_page, $offset]);
		$rows = $wpdb->get_results($wpdb->prepare($sql, $query_values), ARRAY_A);

		// Helper function for sortable column headers
		$sort_link = function($column, $label) use ($orderby, $order) {
			$new_order = ($orderby === $column && $order === 'ASC') ? 'DESC' : 'ASC';
			$arrow = '';
			if ($orderby === $column) {
				$arrow = $order === 'ASC' ? ' ▲' : ' ▼';
			}
			$url = add_query_arg(['orderby'=>$column, 'order'=>$new_order]);
			return '<a href="'.esc_url($url).'" style="text-decoration:none;color:inherit">'.$label.$arrow.'</a>';
		};

		// Display results info
		echo '<p>' . sprintf(__('Showing %d-%d of %d bookings', 'free-short-term-rental-booking'), 
			min($offset + 1, $total_items), 
			min($offset + $per_page, $total_items), 
			$total_items) . '</p>';

		echo '<table class="widefat striped" id="fstrb-bookings-table">';
		echo '<thead><tr>
				<th style="width:50px">'.$sort_link('id', 'ID').'</th>
				<th>'.$sort_link('status', __('Status', 'free-short-term-rental-booking')).'</th>
				<th>'.$sort_link('source', __('Source', 'free-short-term-rental-booking')).'</th>
				<th>'.$sort_link('unit_title', __('Unit', 'free-short-term-rental-booking')).'</th>
				<th>'.$sort_link('date_from', __('From', 'free-short-term-rental-booking')).'</th>
				<th>'.$sort_link('date_to', __('To', 'free-short-term-rental-booking')).'</th>
				<th>'.$sort_link('customer_name', __('Guest', 'free-short-term-rental-booking')).'</th>
				<th>'.__('Contact', 'free-short-term-rental-booking').'</th>
				<th>'.__('Reservation #', 'free-short-term-rental-booking').'</th>
				<th>'.__('Total', 'free-short-term-rental-booking').'</th>
				<th>'.$sort_link('created_at', __('Created', 'free-short-term-rental-booking')).'</th>
				<th>'.__('Actions', 'free-short-term-rental-booking').'</th>
			  </tr></thead>';
		echo '<tbody>';
		
		foreach ($rows as $r) {
			$guestFull = esc_html(trim($r['customer_name'] . ' ' . ($r['customer_lastname'] ?? '')));
			if ($r['customer_gender']) $guestFull .= ' (' . esc_html($r['customer_gender']) . ')';

			$confirmUrl = wp_nonce_url(admin_url('admin.php?page=fstrb&action=confirm&id=' . $r['id']), 'fstrb_booking_action_' . $r['id']);
			$deleteUrl = wp_nonce_url(admin_url('admin.php?page=fstrb&action=delete&id=' . $r['id']), 'fstrb_booking_action_' . $r['id']);
			
			// Source badge
			$source_display = '';
			if (strpos($r['source'], 'ical_') === 0) {
				$provider_key = str_replace('ical_', '', $r['source']);
				$provider_labels = [
					'airbnb' => 'Airbnb',
					'booking' => 'Booking.com',
					'e_chalupy' => 'e-chalupy.cz',
					'megaubytovanie' => 'Megaubytovanie',
					'hauzi' => 'Hauzi',
					'bezrealitky' => 'BezRealitky',
				];
				$provider_colors = [
					'airbnb' => '#FF5A5F',
					'booking' => '#003580',
					'e_chalupy' => '#4CAF50',
					'megaubytovanie' => '#FF9800',
					'hauzi' => '#2196F3',
					'bezrealitky' => '#9C27B0',
				];

				$source_color = $provider_colors[$provider_key] ?? '#666';
			// For unknown providers, display the domain name nicely formatted
			if (!isset($provider_labels[$provider_key])) {
				// If it looks like a domain (contains a dot), display as-is
				// Otherwise capitalize it
				$source_label = strpos($provider_key, '.') !== false ? $provider_key : ucfirst($provider_key);
			} else {
				$source_label = $provider_labels[$provider_key];
			}
			$source_display = '<span style="display:inline-block;padding:3px 8px;background:'.$source_color.';color:#fff;border-radius:3px;font-size:11px;font-weight:bold" title="'.esc_attr($r['source']).'">'.esc_html($source_label).'</span>';
			} else {
				// For direct/admin, show text
				$source_display = esc_html($r['source']);
			}

			echo '<tr>';
			echo '<td>' . esc_html($r['id']) . '</td>';
			echo '<td>' . esc_html($r['status']) . '</td>';
			echo '<td>' . $source_display . '</td>';
			echo '<td>' . esc_html($r['unit_title'] ?: $r['unit_id']) . '</td>';
			echo '<td>' . esc_html($r['date_from']) . '</td>';
			echo '<td>' . esc_html($r['date_to']) . '</td>';
			echo '<td>' . $guestFull . '</td>';
			echo '<td>';
			if ($r['customer_email']) echo esc_html($r['customer_email']) . '<br>';
			if ($r['customer_phone']) echo esc_html($r['customer_phone']);
			
			// Show count of custom fields if any
			$meta_count = (int)$wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM {$t['booking_meta']} WHERE booking_id=%d", $r['id']));
			if ($meta_count > 0) {
				echo '<br><span style="background:#e5f3ff;color:#0073aa;padding:2px 5px;border-radius:3px;font-size:10px;margin-top:5px;display:inline-block">'.sprintf(__('%d custom fields', 'free-short-term-rental-booking'), $meta_count).'</span>';
			}
			echo '</td>';
			echo '<td>';
			if ($r['external_booking_id']) {
				echo '<strong style="color:#0073aa">' . esc_html($r['external_booking_id']) . '</strong>';
			}
			if ($r['ical_summary'] || $r['ical_description'] || $meta_count > 0) {
				echo '<br><a href="#" class="fstrb-toggle-details" data-booking-id="'.$r['id'].'" style="font-size:11px">'.__('Show details', 'free-short-term-rental-booking').'</a>';
			}
			echo '</td>';
			echo '<td>' . ($r['total_price'] ? esc_html($r['total_price'] . ' ' . $r['currency']) : '-') . '</td>';
			echo '<td>' . esc_html(substr($r['created_at'], 0, 16)) . '</td>';
			echo '<td>';
			if ($r['status'] === 'pending') {
				echo '<a href="' . $confirmUrl . '" class="button button-small button-primary">' . __('Confirm', 'free-short-term-rental-booking') . '</a> ';
			}
			echo '<a href="' . $deleteUrl . '" onclick="return confirm(\''.__('Delete this booking?', 'free-short-term-rental-booking').'\')" class="button button-small button-link-delete" style="color:#a00">' . __('Delete', 'free-short-term-rental-booking') . '</a>';
			echo '</td>';
			echo '</tr>';
			
			// Hidden details row
			if ($r['ical_summary'] || $r['ical_description'] || $r['ical_uid'] || $meta_count > 0) {
				echo '<tr class="fstrb-details-row" id="fstrb-details-'.$r['id'].'" style="display:none">';
				echo '<td colspan="12" style="background:#f9f9f9;padding:15px">';
				echo '<strong>' . __('iCal Details:', 'free-short-term-rental-booking') . '</strong><br>';
				if ($r['ical_summary']) echo '<strong>' . __('Summary:', 'free-short-term-rental-booking') . '</strong> ' . esc_html($r['ical_summary']) . '<br>';
				if ($r['ical_description']) {
					$desc = esc_html($r['ical_description']);
					$desc = nl2br($desc);
					echo '<strong>' . __('Description:', 'free-short-term-rental-booking') . '</strong><br><div style="max-height:150px;overflow-y:auto;padding:5px;background:#fff;border:1px solid #ddd">' . $desc . '</div><br>';
				}
				if ($r['ical_uid']) echo '<strong>UID:</strong> <code style="font-size:10px">' . esc_html($r['ical_uid']) . '</code><br>';
				
				// Custom fields
				$meta_rows = $wpdb->get_results($wpdb->prepare("SELECT meta_key, meta_value FROM {$t['booking_meta']} WHERE booking_id=%d", $r['id']), ARRAY_A);
				if ($meta_rows) {
					echo '<div style="margin-top:10px;padding:10px;background:#fff;border:1px solid #eee;border-left:4px solid #0073aa">';
					echo '<strong>' . __('Custom Contact Fields:', 'free-short-term-rental-booking') . '</strong><ul style="margin:5px 0 0 20px;list-style:disc">';
					foreach ($meta_rows as $mr) {
						// Try to get field label for current user language
						$fid = $wpdb->get_var($wpdb->prepare("SELECT id FROM {$t['contact_fields']} WHERE field_key=%s", $mr['meta_key']));
						$label = '';
						if ($fid) {
							$label = $wpdb->get_var($wpdb->prepare("SELECT meta_value FROM {$t['contact_field_meta']} WHERE field_id=%d AND meta_key='label' AND locale=%s", $fid, substr(get_user_locale(), 0, 2)));
							if (!$label) $label = $wpdb->get_var($wpdb->prepare("SELECT meta_value FROM {$t['contact_field_meta']} WHERE field_id=%d AND meta_key='label' AND (locale IS NULL OR locale='')", $fid));
						}
						$display_key = $label ?: $mr['meta_key'];
						echo '<li><strong>'.esc_html($display_key).':</strong> '.esc_html($mr['meta_value']).'</li>';
					}
					echo '</ul></div>';
				}
				echo '</td></tr>';
			}
		}
		
		if (!$rows) echo '<tr><td colspan="12">' . __('No bookings found.', 'free-short-term-rental-booking') . '</td></tr>';
		echo '</tbody></table>';

		// Pagination
		if ($total_pages > 1) {
			echo '<div class="tablenav" style="margin-top:15px">';
			echo '<div class="tablenav-pages">';
			echo '<span class="displaying-num">' . sprintf(__('%s items', 'free-short-term-rental-booking'), number_format_i18n($total_items)) . '</span>';
			
			$page_links = paginate_links([
				'base' => add_query_arg('paged', '%#%'),
				'format' => '',
				'prev_text' => '&laquo;',
				'next_text' => '&raquo;',
				'total' => $total_pages,
				'current' => $paged,
			]);
			echo '<span class="pagination-links">' . $page_links . '</span>';
			echo '</div>';
			
			// Per page selector
			echo '<div style="float:right;margin-top:5px">';
			echo '<label>' . __('Items per page:', 'free-short-term-rental-booking') . ' ';
			echo '<select onchange="window.location.href=this.value">';
			foreach ([25, 50, 100, 200] as $pp) {
				$url = add_query_arg(['per_page'=>$pp, 'paged'=>1]);
				printf('<option value="%s" %s>%d</option>', esc_url($url), selected($per_page, $pp, false), $pp);
			}
			echo '</select></label>';
			echo '</div>';
			echo '</div>';
		}

		// JavaScript for toggle details
		?>
		<script>
		document.addEventListener('DOMContentLoaded', function() {
			document.querySelectorAll('.fstrb-toggle-details').forEach(function(link) {
				link.addEventListener('click', function(e) {
					e.preventDefault();
					var bookingId = this.getAttribute('data-booking-id');
					var detailsRow = document.getElementById('fstrb-details-' + bookingId);
					if (detailsRow.style.display === 'none') {
						detailsRow.style.display = 'table-row';
						this.textContent = '<?php echo esc_js(__('Hide details', 'free-short-term-rental-booking')); ?>';
					} else {
						detailsRow.style.display = 'none';
						this.textContent = '<?php echo esc_js(__('Show details', 'free-short-term-rental-booking')); ?>';
					}
				});
			});
		});
		</script>
		<?php
		echo '</div>';
		self::render_donation_box();
	}

	private static function unit_select($selected=0) {
		$units = get_posts(['post_type'=>'fstrb_unit','numberposts'=>-1,'post_status'=>['publish','draft']]);
		echo '<select name="unit_id">';
		echo '<option value="0">— ' . __('select unit', 'free-short-term-rental-booking') . ' —</option>';
		foreach ($units as $u) {
			printf('<option value="%d" %s>%s</option>', $u->ID, selected($selected,$u->ID,false), esc_html($u->post_title));
		}
		echo '</select>';
	}

	public static function page_general() {
		echo '<div class="wrap"><h1>' . __('General Settings', 'free-short-term-rental-booking') . '</h1>';

		if (isset($_POST['fstrb_save_general'])) {
			check_admin_referer('fstrb_general');
			update_option('fstrb_available_currencies', sanitize_text_field($_POST['fstrb_available_currencies'] ?? 'EUR'));
			echo '<div class="notice notice-success"><p>' . __('Settings saved.', 'free-short-term-rental-booking') . '</p></div>';
		}

		?>
		<form method="post" action="">
			<?php wp_nonce_field('fstrb_general'); ?>
			<table class="form-table">
                <tr>
					<th scope="row"><?php _e('Available Currencies', 'free-short-term-rental-booking'); ?></th>
					<td>
						<input type="text" name="fstrb_available_currencies" value="<?php echo esc_attr(get_option('fstrb_available_currencies', 'EUR')); ?>" class="regular-text">
						<p class="description"><?php _e('Comma separated list, e.g. EUR,CZK', 'free-short-term-rental-booking'); ?></p>
					</td>
				</tr>
			</table>
			<p class="submit">
				<button type="submit" name="fstrb_save_general" class="button button-primary"><?php _e('Save Changes', 'free-short-term-rental-booking'); ?></button>
			</p>
		</form>
		</div>
		<?php
		self::render_donation_box();
	}

	public static function page_calendar() {
		$unit_id = isset($_GET['unit_id']) ? (int)$_GET['unit_id'] : 0;

		echo '<div class="wrap"><h1>' . __('Calendar', 'free-short-term-rental-booking') . '</h1>';
		echo '<form method="get" style="margin:10px 0 20px 0">';
		echo '<input type="hidden" name="page" value="fstrb-calendar" />';
		self::unit_select($unit_id);
		submit_button(__('Show', 'free-short-term-rental-booking'), 'secondary', '', false);
		echo '</form>';

		if (!$unit_id) { 
			echo '</div>'; 
			return; 
		}

		// Render Booking Form (reuses frontend logic)
		// This accepts styling settings from 'Styling' page too.
		$html = FSTRB_Shortcodes::booking_form(['unit' => $unit_id]);
		
		// Inject isAdmin flag for JS
		$data = FSTRB_Shortcodes::get_localization_data();
		$data['isAdmin'] = true;
		wp_localize_script('fstrb-frontend', 'FSTRB', $data);

		echo $html;
		echo '</div>';
		echo '</div>';
		self::render_donation_box();
	}



	public static function page_services() {
		global $wpdb;
		$t = FSTRB_DB::tables();

		echo '<div class="wrap"><h1>' . __('Services', 'free-short-term-rental-booking') . '</h1>';

		// Delete action
		if (isset($_GET['action']) && $_GET['action'] === 'delete' && isset($_GET['id'])) {
			check_admin_referer('fstrb_delete_service');
			$id = (int)$_GET['id'];
			$wpdb->delete($t['services'], ['id' => $id]);
			$wpdb->delete($t['unit_services'], ['service_id' => $id]); // cleanup
			echo '<div class="notice notice-success"><p>' . __('Service deleted.', 'free-short-term-rental-booking') . '</p></div>';
		}

		// Edit/Add logic
		$edit_id = isset($_REQUEST['edit_id']) ? (int)$_REQUEST['edit_id'] : 0;
		$edit_data = null;
		if ($edit_id) {
			$edit_data = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$t['services']} WHERE id=%d", $edit_id));
		}

		// Save service
		if (isset($_POST['fstrb_save_service'])) {
			check_admin_referer('fstrb_services');
			$data = [
				'name' => sanitize_text_field($_POST['name']),
				'code' => sanitize_text_field($_POST['code']),
				'pricing_type' => sanitize_text_field($_POST['pricing_type']),
				'description' => sanitize_textarea_field($_POST['description']),
				'is_active' => !empty($_POST['is_active']) ? 1 : 0,
				'is_optional' => !empty($_POST['is_optional']) ? 1 : 0,
				'sort_order' => (int)($_POST['sort_order'] ?? 0),
			];

			// Currencies for service base prices are stored in metadata
			$currencies = array_map('trim', explode(',', get_option('fstrb_available_currencies', 'EUR')));

			$svc_id = 0;
			if ($edit_id && $edit_data) {
				// Update
				$wpdb->update($t['services'], $data, ['id'=>$edit_id]);
				echo '<div class="notice notice-success"><p>' . __('Updated.', 'free-short-term-rental-booking') . '</p></div>';
				$edit_data = (object)$data; 
				$svc_id = $edit_id;
			} else {
				// Insert
				$wpdb->insert($t['services'], $data);
				$svc_id = $wpdb->insert_id;
				echo '<div class="notice notice-success"><p>' . __('Saved.', 'free-short-term-rental-booking') . '</p></div>';
				// Reset form
				$edit_data = null;
				$edit_id = 0;
			}

			// Save Meta
			if ($svc_id && isset($_POST['meta'])) {
				$wpdb->delete($t['service_meta'], ['service_id' => $svc_id]);
				foreach ($_POST['meta'] as $pair) {
					if (empty($pair['key'])) continue;
					$wpdb->insert($t['service_meta'], [
						'service_id' => $svc_id,
						'meta_key' => sanitize_text_field($pair['key']),
						'meta_value' => sanitize_textarea_field($pair['value']),
						'locale' => !empty($pair['locale']) ? sanitize_text_field($pair['locale']) : null,
					]);
				}
			}

			// Save Currency Prices
			if ($svc_id) {
				foreach ($currencies as $c) {
					$c_price = (float)($_POST['price_' . $c] ?? 0);
					$wpdb->query($wpdb->prepare(
						"DELETE FROM {$t['service_meta']} WHERE service_id=%d AND meta_key=%s",
						$svc_id, 'price_' . $c
					));
					$wpdb->insert($t['service_meta'], [
						'service_id' => $svc_id,
						'meta_key' => 'price_' . $c,
						'meta_value' => $c_price,
					]);
				}
			}
		}

		// Quick Add Scripts
		?>
		<script>
		function fillService(name, code, type, price) {
			document.getElementById('svc_name').value = name;
			document.getElementById('svc_code').value = code;
			document.getElementById('svc_type').value = type;
			document.getElementById('svc_price').value = price;
		}
		</script>
		<div style="margin-bottom:20px;background:#fff;padding:10px;border:1px solid #ccd0d4">
			<strong><?php _e('Quick Add:', 'free-short-term-rental-booking'); ?></strong>
			<button type="button" class="button" onclick="fillService('Cleaning','CLEANING','per_stay',0)"><?php _e('Cleaning', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Bed Linen','LINEN','per_stay',0)"><?php _e('Bed Linen', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Towels','TOWELS','per_stay',0)"><?php _e('Towels', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Parking','PARKING','per_stay',0)"><?php _e('Parking', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Garage','GARAGE','per_stay',0)"><?php _e('Garage', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Breakfast','BREAKFAST','per_person_night',0)"><?php _e('Breakfast', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Half Board','HALF_BOARD','per_person_night',0)"><?php _e('Half Board', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('All Inclusive','ALL_INCLUSIVE','per_person_night',0)"><?php _e('All Inclusive', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Netflix / Streaming','NETFLIX','per_stay',0)"><?php _e('Netflix / Streaming', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Game Console','CONSOLE','per_stay',0)"><?php _e('Game Console', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Sauna','SAUNA','per_stay',0)"><?php _e('Sauna', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Wellness / Spa','WELLNESS','per_person',0)"><?php _e('Wellness / Spa', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Pool','POOL','per_stay',0)"><?php _e('Pool', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Late Check-out','LATE_CHECKOUT','per_stay',0)"><?php _e('Late Check-out', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Pet Fee','PET_FEE','per_stay',0)"><?php _e('Pet Fee', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Bike Rental','BIKE_RENTAL','per_night',0)"><?php _e('Bike Rental', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('EV Charging','EV_CHARGE','per_stay',0)"><?php _e('EV Charging', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Wood for Fireplace','FIREWOOD','per_stay',0)"><?php _e('Wood for Fireplace', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Airport Shuttle','SHUTTLE','per_stay',0)"><?php _e('Airport Shuttle', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Extra Bed','EXTRA_BED','per_night',0)"><?php _e('Extra Bed', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Tourist Tax','TOURIST_TAX','per_person_night',0)"><?php _e('Tourist Tax', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Air Conditioning','AIR_CONDITIONING','per_stay',0)"><?php _e('Air Conditioning', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('TV','TV','per_stay',0)"><?php _e('TV', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('WIFI','WIFI','per_stay',0)"><?php _e('WIFI', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Boat Rental','BOAT_RENTAL','per_stay',0)"><?php _e('Boat Rental', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Paddleboard Rental','PADDLEBOARD_RENTAL','per_stay',0)"><?php _e('Paddleboard Rental', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Beach Access','BEACH_ACCESS','per_stay',0)"><?php _e('Beach Access', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Sunbed & Umbrella','BEACH_SET','per_stay',0)"><?php _e('Sunbed & Umbrella', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Energy','ENERGY','per_stay',0)"><?php _e('Energy', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Car Rental','CAR_RENTAL','per_stay',0)"><?php _e('Car Rental', 'free-short-term-rental-booking'); ?></button>
			<button type="button" class="button" onclick="fillService('Motorcycle Rental','MOTORCYCLE_RENTAL','per_stay',0)"><?php _e('Motorcycle Rental', 'free-short-term-rental-booking'); ?></button>

		</div>
		<?php

		// Form
		$lbl = $edit_id ? __('Edit Service', 'free-short-term-rental-booking') : __('Add Service', 'free-short-term-rental-booking');
		$btn = $edit_id ? __('Update', 'free-short-term-rental-booking') : __('Add', 'free-short-term-rental-booking');
		$val_name = $edit_data ? $edit_data->name : '';
		$val_code = $edit_data ? $edit_data->code : '';
		$val_type = $edit_data ? $edit_data->pricing_type : 'per_stay';
		$val_price = $edit_data ? $edit_data->price : '0';
		$val_sort = $edit_data ? $edit_data->sort_order : '0';
		$val_active = (!$edit_data || $edit_data->is_active) ? 'checked' : '';
		$val_optional = (!$edit_data || $edit_data->is_optional) ? 'checked' : '';

		echo '<form method="post">';
		wp_nonce_field('fstrb_services');
		echo '<input type="hidden" name="edit_id" value="'.$edit_id.'">';
		echo '<h2>' . $lbl . ' ' . ($edit_id ? '<a href="admin.php?page=fstrb-services" class="page-title-action">'.__('Cancel', 'free-short-term-rental-booking').'</a>' : '') . '</h2>';
		echo '<p style="display:flex;gap:10px;align-items:center;flex-wrap:wrap">';
		echo '<label>' . __('Name', 'free-short-term-rental-booking') . ' <br><input id="svc_name" name="name" required value="'.esc_attr($val_name).'"></label>';
		echo '<label>' . __('Code', 'free-short-term-rental-booking') . ' <br><input id="svc_code" name="code" required value="'.esc_attr($val_code).'" placeholder="CODE"></label>';
		echo '<label>' . __('Type', 'free-short-term-rental-booking') . ' <br><select id="svc_type" name="pricing_type">';
		$types = FSTRB_CPT::get_service_pricing_types();
		foreach ($types as $k=>$v) echo '<option value="'.$k.'" '.selected($val_type,$k,false).'>'.$v.'</option>';
		echo '</select></label>';

		echo '<label>' . __('Order', 'free-short-term-rental-booking') . ' <br><input type="number" name="sort_order" value="'.esc_attr($val_sort).'" style="width:80px"></label>';
		
		$currencies = array_map('trim', explode(',', get_option('fstrb_available_currencies', 'EUR')));
		foreach ($currencies as $c) {
			$c_val = $edit_id ? $wpdb->get_var($wpdb->prepare("SELECT meta_value FROM {$t['service_meta']} WHERE service_id=%d AND meta_key=%s", $edit_id, 'price_' . $c)) : '0';
			echo '<label>' . sprintf(__('Price (%s)', 'free-short-term-rental-booking'), $c) . ' <br><input type="number" step="0.01" name="price_'.$c.'" value="'.esc_attr($c_val).'"></label>';
		}

		echo '<label>' . __('Description', 'free-short-term-rental-booking') . ' <br><textarea name="description" rows="3" style="width:100%;max-width:400px">'.esc_textarea($edit_data->description ?? '').'</textarea></label>';
		
		// Meta Data Builder
		$meta = [];
		if ($edit_id) {
			$meta_rows = $wpdb->get_results($wpdb->prepare("SELECT meta_key, meta_value, locale FROM {$t['service_meta']} WHERE service_id=%d", $edit_id), ARRAY_A);
			foreach ($meta_rows as $mr) {
				$meta[] = ['key' => $mr['meta_key'], 'value' => $mr['meta_value'], 'locale' => $mr['locale']];
			}
		}

		$locales = ['' => __('All (Global)', 'free-short-term-rental-booking'), 'cs' => 'CZ', 'en' => 'EN', 'de' => 'DE', 'pl' => 'PL'];
		?>
		<div style="width:100%;border-top:1px solid #ccc;margin-top:10px;padding-top:10px">
			<strong><?php _e('Metadata (Key-Value)', 'free-short-term-rental-booking'); ?></strong>
			<div id="fstrb_meta_wrapper">
				<?php foreach ($meta as $idx => $pair): ?>
				<div class="fstrb-meta-row" style="margin-bottom:5px;display:flex;gap:5px">
					<select name="meta[<?php echo $idx; ?>][locale]" style="width:80px">
						<?php foreach ($locales as $k=>$v) echo '<option value="'.$k.'" '.selected($pair['locale'] ?? '', $k, false).'>'.$v.'</option>'; ?>
					</select>
					<input type="text" name="meta[<?php echo $idx; ?>][key]" placeholder="Key" value="<?php echo esc_attr($pair['key'] ?? ''); ?>" size="15">
					<input type="text" name="meta[<?php echo $idx; ?>][value]" placeholder="Value" value="<?php echo esc_attr($pair['value'] ?? ''); ?>" size="30">
					<button type="button" class="button button-small" onclick="this.parentNode.remove()">x</button>
				</div>
				<?php endforeach; ?>
			</div>
			<button type="button" class="button button-small" onclick="addMetaRow()"><?php _e('+ Add Meta', 'free-short-term-rental-booking'); ?></button>
			<script>
			function addMetaRow() {
				const w = document.getElementById('fstrb_meta_wrapper');
				const idx = w.children.length + new Date().getTime();
				const div = document.createElement('div');
				div.className = 'fstrb-meta-row';
				div.style = 'margin-bottom:5px;display:flex;gap:5px';
				div.innerHTML = `<select name="meta[${idx}][locale]" style="width:80px">
					<option value=""><?php _e('All', 'free-short-term-rental-booking'); ?></option>
					<option value="cs">CZ</option>
					<option value="en">EN</option>
					<option value="de">DE</option>
					<option value="pl">PL</option>
				</select> <input type="text" name="meta[${idx}][key]" placeholder="Key" size="15"> <input type="text" name="meta[${idx}][value]" placeholder="Value" size="30"> <button type="button" class="button button-small" onclick="this.parentNode.remove()">x</button>`;
				w.appendChild(div);
			}
			</script>
		</div>
		<?php

		echo '<label><br><input type="checkbox" name="is_optional" '.$val_optional.'> ' . __('Optional (default)', 'free-short-term-rental-booking') . '</label>';
		echo '<label><br><input type="checkbox" name="is_active" '.$val_active.'> ' . __('active', 'free-short-term-rental-booking') . '</label>';
		echo '<label><br><button class="button button-primary" name="fstrb_save_service" value="1">' . $btn . '</button></label>';
		echo '</p>';
		echo '</form>';

		// List
		$services = $wpdb->get_results("SELECT * FROM {$t['services']} ORDER BY sort_order ASC, name ASC", ARRAY_A);
		echo '<h2>' . __('List', 'free-short-term-rental-booking') . '</h2>';
		echo '<table class="widefat striped"><thead><tr><th>ID</th><th>' . __('Name', 'free-short-term-rental-booking') . '</th><th>' . __('Code', 'free-short-term-rental-booking') . '</th><th>' . __('Type', 'free-short-term-rental-booking') . '</th><th>' . __('Price', 'free-short-term-rental-booking') . '</th><th>' . __('Order', 'free-short-term-rental-booking') . '</th><th>' . __('Optional', 'free-short-term-rental-booking') . '</th><th>' . __('Active', 'free-short-term-rental-booking') . '</th><th>' . __('Actions', 'free-short-term-rental-booking') . '</th></tr></thead><tbody>';
		foreach ($services as $s) {
			$del_url = wp_nonce_url("admin.php?page=fstrb-services&action=delete&id=".$s['id'], 'fstrb_delete_service');
			$edit_url = "admin.php?page=fstrb-services&edit_id=".$s['id'];
			echo '<tr>
				<td>'.esc_html($s['id']).'</td>
				<td>'.esc_html(__($s['name'], 'free-short-term-rental-booking')).'</td>
				<td><code>'.esc_html($s['code']).'</code></td>
				<td>'.esc_html($types[$s['pricing_type']] ?? $s['pricing_type']).'</td>
				<td>'.esc_html($s['price']).'</td>
				<td>'.esc_html($s['sort_order']).'</td>
				<td>'.($s['is_optional']?__('yes', 'free-short-term-rental-booking'):__('no', 'free-short-term-rental-booking')).'</td>
				<td>'.($s['is_active']?__('yes', 'free-short-term-rental-booking'):__('no', 'free-short-term-rental-booking')).'</td>
				<td>
					<a href="'.$edit_url.'" class="button button-small">'.__('Edit', 'free-short-term-rental-booking').'</a>
					<a href="'.$del_url.'" class="button button-small button-link-delete" onclick="return confirm(\''.esc_js(__('Are you sure?', 'free-short-term-rental-booking')).'\')">'.__('Delete', 'free-short-term-rental-booking').'</a>
				</td>
			</tr>';
		}
		echo '</tbody></table></div>';
		self::render_donation_box();
	}

	public static function page_rates() {
		global $wpdb;
		$t = FSTRB_DB::tables();

		// Self-healing: Check tables & columns
		if ($wpdb->get_var("SHOW TABLES LIKE '{$t['rates']}'") != $t['rates']) {
			FSTRB_DB::migrate();
		} else {
			 $col = $wpdb->get_results("SHOW COLUMNS FROM {$t['rates']} LIKE 'operation'");
			 if (empty($col)) {
				 FSTRB_DB::migrate();
			 }
		}

		echo '<div class="wrap"><h1>' . __('Rates & Discounts (Seasonal)', 'free-short-term-rental-booking') . '</h1>';

		// Delete Action
		if (isset($_GET['action']) && $_GET['action'] === 'delete' && isset($_GET['id'])) {
			check_admin_referer('fstrb_delete_rate');
			$id = (int)$_GET['id'];
			
			// Get the rate to identify its group
			$row = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$t['rates']} WHERE id=%d", $id));
			if ($row) {
				// Delete the rate and its "siblings" (same definition, different currency)
				$wpdb->query($wpdb->prepare(
					"DELETE FROM {$t['rates']} WHERE unit_id=%d AND name=%s AND date_from=%s AND date_to=%s AND rule=%s AND operation=%s",
					$row->unit_id, $row->name, $row->date_from, $row->date_to, $row->rule, $row->operation
				));
				echo '<div class="notice notice-success"><p>' . __('Rate group deleted.', 'free-short-term-rental-booking') . '</p></div>';
			}
		}

		$avail_currencies = array_map('trim', explode(',', get_option('fstrb_available_currencies', 'EUR')));

		// Edit Data
		$edit_id = isset($_GET['edit_id']) ? (int)$_GET['edit_id'] : 0;
		$edit_data = null;
		$sibling_map = []; // currency => id

		if ($edit_id) {
			$edit_data = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$t['rates']} WHERE id=%d", $edit_id));
			if ($edit_data) {
				// Find siblings (same unit, name, dates, rule, operation) to populate other currencies
				$siblings = $wpdb->get_results($wpdb->prepare(
					"SELECT id, currency, price FROM {$t['rates']} 
					 WHERE unit_id=%d AND name=%s AND date_from=%s AND date_to=%s AND rule=%s AND operation=%s",
					$edit_data->unit_id, $edit_data->name, $edit_data->date_from, $edit_data->date_to, $edit_data->rule, $edit_data->operation
				));
				if ($siblings && is_array($siblings)) {
					foreach ($siblings as $sib) {
						$sibling_map[$sib->currency] = $sib;
					}
				}
			}
		}

		// Save Action
		if (isset($_POST['fstrb_save_rate'])) {
			check_admin_referer('fstrb_rates');
			
			$common_data = [
				'unit_id' => (int)($_POST['unit_id'] ?? 0),
				'name' => sanitize_text_field($_POST['name'] ?? ''),
				'date_from' => sanitize_text_field($_POST['date_from'] ?? ''),
				'date_to' => sanitize_text_field($_POST['date_to'] ?? ''),
				'rule' => sanitize_text_field($_POST['rule'] ?? 'per_unit_night'),
				'operation' => sanitize_text_field($_POST['operation'] ?? 'replace'),
				'min_nights' => !empty($_POST['min_nights']) ? (int)$_POST['min_nights'] : null,
				'priority' => (int)($_POST['priority'] ?? 100),
			];

			// 1. If edit_id is set, determine target group via original data
			$targets = [];
			 if ($edit_id) {
				$orig = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$t['rates']} WHERE id=%d", $edit_id));
				if ($orig) {
					 $sibs = $wpdb->get_results($wpdb->prepare(
						"SELECT id, currency FROM {$t['rates']} 
						 WHERE unit_id=%d AND name=%s AND date_from=%s AND date_to=%s AND rule=%s AND operation=%s",
						$orig->unit_id, $orig->name, $orig->date_from, $orig->date_to, $orig->rule, $orig->operation
					));
					if ($sibs && is_array($sibs)) {
						foreach ($sibs as $s) $targets[$s->currency] = $s->id;
					}
				}
			}

			foreach ($avail_currencies as $curr) {
				$price_val = (float)($_POST['price_' . $curr] ?? 0);
				
				$row_data = array_merge($common_data, [
					'price' => $price_val,
					'currency' => $curr
				]);

				if (isset($targets[$curr])) {
					$wpdb->update($t['rates'], $row_data, ['id' => $targets[$curr]]);
					unset($targets[$curr]);
				} else {
					$wpdb->insert($t['rates'], $row_data);
				}
			}
			
			echo '<div class="notice notice-success"><p>' . __('Saved.', 'free-short-term-rental-booking') . '</p></div>';
			
			// Clear edit mode
			$edit_id = 0; 
			$edit_data = null;
			$sibling_map = [];
		}

		// Form Defaults
		$val_unit = $edit_data ? $edit_data->unit_id : (int)($_GET['unit_id'] ?? 0);
		$val_name = $edit_data ? $edit_data->name : '';
		$val_from = $edit_data ? $edit_data->date_from : '';
		$val_to   = $edit_data ? $edit_data->date_to : '';
		$val_rule = $edit_data ? $edit_data->rule : 'per_unit_night';
		$val_op   = $edit_data ? ($edit_data->operation ?? 'replace') : 'replace';
		$val_prio = $edit_data ? $edit_data->priority : 100;

		$lbl = $edit_id ? __('Edit Rate', 'free-short-term-rental-booking') : __('Add Rate/Discount', 'free-short-term-rental-booking');
		$btn = $edit_id ? __('Update All Currencies', 'free-short-term-rental-booking') : __('Add', 'free-short-term-rental-booking');

		echo '<form method="post">';
		wp_nonce_field('fstrb_rates');
		echo '<h2>' . $lbl . ($edit_id ? ' <a href="admin.php?page=fstrb-rates" class="page-title-action">'.__('Cancel', 'free-short-term-rental-booking').'</a>' : '') . '</h2>';
		
		echo '<div style="background:#fff;padding:15px;border:1px solid #ccd0d4;max-width:800px">';
		
		echo '<p><label><strong>' . __('Unit', 'free-short-term-rental-booking') . '</strong><br>';
		self::unit_select($val_unit);
		echo '</label></p>';

		echo '<p><label><strong>' . __('Name', 'free-short-term-rental-booking') . '</strong> (e.g. Summer Season, Long Stay Discount)<br><input name="name" required value="'.esc_attr($val_name).'" style="width:100%"></label></p>';

		echo '<div style="display:flex;gap:10px;flex-wrap:wrap">';
		echo '<div><label><strong>' . __('From', 'free-short-term-rental-booking') . '</strong><br><input type="date" name="date_from" required value="'.esc_attr($val_from).'"></label></div>';
		echo '<div><label><strong>' . __('To', 'free-short-term-rental-booking') . '</strong><br><input type="date" name="date_to" required value="'.esc_attr($val_to).'"></label></div>';
		echo '</div>';

		echo '<p><strong>' . __('Type & Value', 'free-short-term-rental-booking') . '</strong></p>';
		echo '<div style="display:flex;gap:10px;flex-wrap:wrap;align-items:flex-end;margin-bottom:10px">';
		
		echo '<div><label>' . __('Operation', 'free-short-term-rental-booking') . '<br>';
		echo '<select name="operation">';
		echo '<option value="replace" '.selected($val_op,'replace',false).'>'.__('Price Override (Standard)', 'free-short-term-rental-booking').'</option>';
		echo '<option value="discount_percent" '.selected($val_op,'discount_percent',false).'>'.__('Discount (%)', 'free-short-term-rental-booking').'</option>';
		echo '<option value="discount_amount" '.selected($val_op,'discount_amount',false).'>'.__('Discount (Fixed Amount)', 'free-short-term-rental-booking').'</option>';
		echo '</select></label></div>';
		echo '</div>'; // end op row

		echo '<div style="background:#f9f9f9;padding:10px;border:1px solid #eee;margin-bottom:15px">';
		echo '<strong>' . __('Values per Currency:', 'free-short-term-rental-booking') . '</strong><br>';
		echo '<div style="display:flex;gap:15px;flex-wrap:wrap;margin-top:5px">';
		foreach ($avail_currencies as $curr) {
			$val = '';
			if (isset($sibling_map[$curr])) {
				$val = $sibling_map[$curr]->price;
			}
			echo '<div><label>' . esc_html($curr) . '<br><input type="number" step="0.01" name="price_'.esc_attr($curr).'" value="'.esc_attr($val).'" placeholder="0.00" style="width:100px"></label></div>';
		}
		echo '</div></div>';

		echo '<p><strong>' . __('Application Rule', 'free-short-term-rental-booking') . '</strong> <span class="description">(' . sprintf(__('Most discounts should use %s or %s', 'free-short-term-rental-booking'), __('per unit/night', 'free-short-term-rental-booking'), __('weekend', 'free-short-term-rental-booking')) . ')</span></p>';
		echo '<div style="display:flex;gap:10px;align-items:center">';
		echo '<select name="rule">';
		$rules = [
			'per_unit_night' => __('Every Night (per unit)', 'free-short-term-rental-booking'),
			'per_person_night' => __('Every Night (per person)', 'free-short-term-rental-booking'),
			'weekend' => __('Weekends Only (Fri/Sat/Sun)', 'free-short-term-rental-booking'),
			'workday' => __('Workdays Only (Mon-Thu)', 'free-short-term-rental-booking'),
			'christmas' => __('Christmas (Dec 24-26)', 'free-short-term-rental-booking'),
			'new_year' => __('New Year (Dec 31)', 'free-short-term-rental-booking'),
			'winter' => __('Winter Season (Dec-Mar)', 'free-short-term-rental-booking'),
			'summer' => __('Summer Season (Jul-Aug)', 'free-short-term-rental-booking'),
			'off_season' => __('Off-season (Apr-Jun, Sep-Nov)', 'free-short-term-rental-booking'),
			'fixed' => __('One-time Fixed (legacy)', 'free-short-term-rental-booking'),
		];
		foreach ($rules as $k=>$v) echo '<option value="'.$k.'" '.selected($val_rule, $k, false).'>'.$v.'</option>';
		echo '</select>';

		echo '<label>' . __('Priority', 'free-short-term-rental-booking') . ' <input type="number" name="priority" value="'.esc_attr($val_prio).'" style="width:60px"> <span class="description">'.__('(Lower = applied first)', 'free-short-term-rental-booking').'</span></label>';
		echo '</div>';

		echo '<p><button class="button button-primary" name="fstrb_save_rate" value="1">' . $btn . '</button>';
		if ($edit_id) echo ' <a href="admin.php?page=fstrb-rates" class="button">'.__('Cancel', 'free-short-term-rental-booking').'</a>';
		echo '</p>';
		
		echo '</div>'; // end container
		echo '</form>';

		// List
		$rates = $wpdb->get_results("SELECT r.*, p.post_title FROM {$t['rates']} r LEFT JOIN {$wpdb->posts} p ON p.ID=r.unit_id ORDER BY r.unit_id ASC, r.date_from DESC, r.name ASC", ARRAY_A);
		
		// Group rates
		$groups = [];
		if ($rates && is_array($rates)) {
			foreach ($rates as $r) {
				$key = implode('|', [$r['unit_id'], $r['name'], $r['date_from'], $r['date_to'], $r['rule'], ($r['operation'] ?? 'replace'), $r['priority'], ($r['min_nights'] ?? 0)]);
				if (!isset($groups[$key])) {
					$groups[$key] = $r; // keep main info
					$groups[$key]['price_html'] = [];
				}
				
				$val_display = esc_html($r['price']);
				if (($r['operation'] ?? '') === 'discount_percent') $val_display .= '% ' . esc_html($r['currency']);
				else $val_display .= ' ' . esc_html($r['currency']);
				
				$groups[$key]['price_html'][] = $val_display;
				$groups[$key]['id'] = $r['id']; // Use representative ID
			}
		}

		echo '<h2>' . __('List', 'free-short-term-rental-booking') . '</h2>';
		echo '<table class="widefat striped"><thead><tr>
				<th>'.__('Unit', 'free-short-term-rental-booking').'</th>
				<th>' . __('Name', 'free-short-term-rental-booking') . '</th>
				<th>' . __('Dates', 'free-short-term-rental-booking') . '</th>
				<th>' . __('Type', 'free-short-term-rental-booking') . '</th>
				<th>' . __('Values', 'free-short-term-rental-booking') . '</th>
				<th>' . __('Rule', 'free-short-term-rental-booking') . '</th>
				<th>' . __('Prio', 'free-short-term-rental-booking') . '</th>
				<th>' . __('Actions', 'free-short-term-rental-booking') . '</th>
			  </tr></thead><tbody>';
		
		foreach ($groups as $g) {
			$del_url = wp_nonce_url("admin.php?page=fstrb-rates&action=delete&id=".$g['id'], 'fstrb_delete_rate');
			$edit_url = "admin.php?page=fstrb-rates&edit_id=".$g['id'];

			$op_label = $g['operation'] ?? 'replace';
			if ($op_label === 'replace') $op_label = __('Override', 'free-short-term-rental-booking');
			elseif ($op_label === 'discount_percent') $op_label = __('Discount %', 'free-short-term-rental-booking');
			elseif ($op_label === 'discount_amount') $op_label = __('Discount $', 'free-short-term-rental-booking');

			echo '<tr>
				<td>'.esc_html($g['post_title'] ?: $g['unit_id']).'</td>
				<td><strong>'.esc_html(__($g['name'], 'free-short-term-rental-booking')).'</strong></td>
				<td>'.esc_html($g['date_from']) . ' &rarr; ' . esc_html($g['date_to']).'</td>
				<td>'.esc_html($op_label).'</td>
				<td>'.implode('<br>', $g['price_html']).'</td>
				<td>'.esc_html($g['rule']).'</td>
				<td>'.esc_html($g['priority']).'</td>
				<td>
					<a href="'.$edit_url.'" class="button button-small">'.__('Edit', 'free-short-term-rental-booking').'</a>
					<a href="'.$del_url.'" class="button button-small button-link-delete" onclick="return confirm(\''.esc_js(__('Delete group?', 'free-short-term-rental-booking')).'\')">'.__('Delete', 'free-short-term-rental-booking').'</a>
				</td>
			</tr>';
		}
		echo '</tbody></table></div>';
		self::render_donation_box();
	}

	public static function page_styling() {
		echo '<div class="wrap"><h1>' . __('Short Form Styling', 'free-short-term-rental-booking') . '</h1>';

		// Save
		if (isset($_POST['fstrb_save_styling'])) {
			check_admin_referer('fstrb_styling');
			
			$mode = sanitize_text_field($_POST['fstrb_styling_mode'] ?? 'off');
			$css_url = esc_url_raw($_POST['fstrb_custom_css_url'] ?? '');


			$fields = [
				'font_family', 'cal_bg', 'cal_text', 'cal_active', 'cal_occupied', 'form_bg', 'max_width', 'font_size', 'svc_label', 'meta_style', 'meta_sep',
				'cal_h_text', 'cal_h_size', 'cal_h_color', 'cal_h_weight', 'cal_h_align',
				'frm_h_text', 'frm_h_size', 'frm_h_color', 'frm_h_weight', 'frm_h_align',
				'cal_occ_text', 'cal_sel_text', 'cal_outline', 'cal_outline_size',
				'bs_color', 'bs_blur', 'bs_spread', 'bs_x', 'bs_y',
				'cal_date_size', 'cal_date_color', 'cal_date_weight',
				'cal_lbl_size', 'cal_lbl_color', 'cal_lbl_weight',
				'btn_text', 'btn_bg', 'btn_color', 'btn_size', 'btn_radius', 'btn_font', 'btn_weight'
			];
			$visual = [];
			foreach ($fields as $f) {
				$val = $_POST['vis_' . $f] ?? '';
				if (in_array($f, ['cal_bg', 'cal_text', 'cal_active', 'cal_occupied', 'form_bg', 'cal_h_color', 'frm_h_color', 'cal_occ_text', 'cal_sel_text', 'cal_outline', 'bs_color', 'cal_date_color', 'cal_lbl_color', 'btn_bg', 'btn_color'])) {
					$visual[$f] = sanitize_hex_color($val);
				} elseif (in_array($f, ['max_width', 'font_size', 'cal_h_size', 'frm_h_size', 'cal_outline_size', 'bs_blur', 'bs_spread', 'bs_x', 'bs_y', 'cal_date_size', 'cal_lbl_size', 'btn_size', 'btn_radius'])) {
					$visual[$f] = (int)$val;
				} else {
					$visual[$f] = sanitize_text_field($val);
				}
				$visual[$f . '_en'] = isset($_POST['vis_' . $f . '_en']) ? 1 : 0;
			}
			$visual['max_width_unit'] = sanitize_text_field($_POST['vis_max_width_unit'] ?? 'px');

			update_option('fstrb_styling_mode', $mode);
			update_option('fstrb_custom_css_url', $css_url);
			update_option('fstrb_visual_settings', $visual);

			echo '<div class="notice notice-success"><p>' . __('Settings saved.', 'free-short-term-rental-booking') . '</p></div>';
		}

		// Retrieve
		$mode = get_option('fstrb_styling_mode', 'off');
		$css_url = get_option('fstrb_custom_css_url', '');
		$visual = get_option('fstrb_visual_settings', []);
		
		$v_font = $visual['font_family'] ?? '';
		$v_bg = $visual['cal_bg'] ?? '#f0f0f0';
		$v_text = $visual['cal_text'] ?? '#333333';
		$v_active = $visual['cal_active'] ?? '#007cba';
		$v_occupied = $visual['cal_occupied'] ?? '#ffdddd';
		$v_form_bg = $visual['form_bg'] ?? '#ffffff';
		$v_width = $visual['max_width'] ?? 400;
		$v_width_unit = $visual['max_width_unit'] ?? 'px';
		$v_size = $visual['font_size'] ?? 14;
		$v_svc_label = $visual['svc_label'] ?? '';
		$v_meta_style = $visual['meta_style'] ?? 'list';
		$v_meta_sep = $visual['meta_sep'] ?? '';

		$en = function($key, $default = false) use ($visual) {
			if (!isset($visual[$key . '_en'])) return $default ? 'checked' : '';
			return (!empty($visual[$key . '_en'])) ? 'checked' : '';
		};

		?>
		<form method="post" action="">
			<?php wp_nonce_field('fstrb_styling'); ?>
			
			<table class="form-table">
				<tr>
					<th scope="row"><?php _e('Styling Mode', 'free-short-term-rental-booking'); ?></th>
					<td>
						<fieldset>
							<label><input type="radio" name="fstrb_styling_mode" value="off" <?php checked($mode, 'off'); ?>> <?php _e('Default (Theme/Plugin CSS)', 'free-short-term-rental-booking'); ?></label><br>
							<label><input type="radio" name="fstrb_styling_mode" value="css" <?php checked($mode, 'css'); ?>> <?php _e('Custom CSS URL', 'free-short-term-rental-booking'); ?></label>
						</fieldset>
					</td>
				</tr>
			</table>

			<hr>

			<h3><?php _e('Custom CSS', 'free-short-term-rental-booking'); ?></h3>
			<table class="form-table">
				<tr>
					<th scope="row"><?php _e('CSS File URL', 'free-short-term-rental-booking'); ?></th>
					<td><input type="url" name="fstrb_custom_css_url" value="<?php echo esc_attr($css_url); ?>" class="regular-text"></td>
				</tr>
			</table>

			<hr>

			<div style="display:flex; justify-content: space-between; align-items: center;">
				<h3><?php _e('Visual Settings', 'free-short-term-rental-booking'); ?></h3>
				<label><input type="checkbox" id="fstrb_toggle_all_styles"> <strong><?php _e('Select All / Deselect All', 'free-short-term-rental-booking'); ?></strong></label>
			</div>

			<table class="form-table fstrb-styling-table">
				<tr>
					<th scope="row">
						<label><input type="checkbox" name="vis_font_family_en" class="fstrb-style-en" <?php echo $en('font_family'); ?>> <?php _e('Font Family', 'free-short-term-rental-booking'); ?></label>
					</th>
					<td>
						<input type="text" name="vis_font_family" value="<?php echo esc_attr($v_font); ?>" placeholder="e.g. Arial, sans-serif" class="regular-text">
						<p class="description"><?php _e('Enter font stack, e.g. "Open Sans", sans-serif', 'free-short-term-rental-booking'); ?></p>
					</td>
				</tr>
				<tr>
					<th scope="row">
						<label><input type="checkbox" name="vis_form_bg_en" class="fstrb-style-en" <?php echo $en('form_bg'); ?>> <?php _e('Form Background', 'free-short-term-rental-booking'); ?></label>
					</th>
					<td><input type="color" name="vis_form_bg" value="<?php echo esc_attr($v_form_bg); ?>"> <span class="description"><?php _e('Background of the booking dialog/form', 'free-short-term-rental-booking'); ?></span></td>
				</tr>
				<tr>
					<th scope="row">
						<label><input type="checkbox" name="vis_cal_bg_en" class="fstrb-style-en" <?php echo $en('cal_bg'); ?>> <?php _e('Calendar Background', 'free-short-term-rental-booking'); ?></label>
					</th>
					<td><input type="color" name="vis_cal_bg" value="<?php echo esc_attr($v_bg); ?>"></td>
				</tr>
				<tr>
					<th scope="row">
						<label><input type="checkbox" name="vis_cal_text_en" class="fstrb-style-en" <?php echo $en('cal_text'); ?>> <?php _e('Calendar Text Color', 'free-short-term-rental-booking'); ?></label>
					</th>
					<td><input type="color" name="vis_cal_text" value="<?php echo esc_attr($v_text); ?>"></td>
				</tr>
				<tr>
					<th scope="row">
						<label><input type="checkbox" name="vis_cal_active_en" class="fstrb-style-en" <?php echo $en('cal_active'); ?>> <?php _e('Calendar Active/Selection', 'free-short-term-rental-booking'); ?></label>
					</th>
					<td><input type="color" name="vis_cal_active" value="<?php echo esc_attr($v_active); ?>"></td>
				</tr>
				<tr>
					<th scope="row">
						<label><input type="checkbox" name="vis_cal_occupied_en" class="fstrb-style-en" <?php echo $en('cal_occupied'); ?>> <?php _e('Calendar Occupied Color', 'free-short-term-rental-booking'); ?></label>
					</th>
					<td><input type="color" name="vis_cal_occupied" value="<?php echo esc_attr($v_occupied); ?>"></td>
				</tr>
				<tr>
					<th scope="row">
						<label><input type="checkbox" name="vis_max_width_en" class="fstrb-style-en" <?php echo $en('max_width'); ?>> <?php _e('Max Width', 'free-short-term-rental-booking'); ?></label>
					</th>
					<td>
						<input type="number" name="vis_max_width" value="<?php echo esc_attr($v_width); ?>" class="small-text"> 
						<select name="vis_max_width_unit">
							<option value="px" <?php selected($v_width_unit, 'px'); ?>>px</option>
							<option value="%" <?php selected($v_width_unit, '%'); ?>>%</option>
						</select>
					</td>
				</tr>
				<tr>
					<th scope="row">
						<label><input type="checkbox" name="vis_font_size_en" class="fstrb-style-en" <?php echo $en('font_size'); ?>> <?php _e('Base Font Size (px)', 'free-short-term-rental-booking'); ?></label>
					</th>
					<td><input type="number" name="vis_font_size" value="<?php echo esc_attr($v_size); ?>" class="small-text"> px</td>
				</tr>
				<tr>
					<th scope="row">
						<label><input type="checkbox" name="vis_cal_occ_text_en" class="fstrb-style-en" <?php echo $en('cal_occ_text'); ?>> <?php _e('Occupied Text Color', 'free-short-term-rental-booking'); ?></label>
					</th>
					<td><input type="color" name="vis_cal_occ_text" value="<?php echo esc_attr($visual['cal_occ_text'] ?? '#ffffff'); ?>"></td>
				</tr>
				<tr>
					<th scope="row">
						<label><input type="checkbox" name="vis_cal_sel_text_en" class="fstrb-style-en" <?php echo $en('cal_sel_text'); ?>> <?php _e('Selected Text Color', 'free-short-term-rental-booking'); ?></label>
					</th>
					<td><input type="color" name="vis_cal_sel_text" value="<?php echo esc_attr($visual['cal_sel_text'] ?? '#ffffff'); ?>"></td>
				</tr>
				<tr>
					<th scope="row">
						<label><input type="checkbox" name="vis_cal_outline_en" class="fstrb-style-en" <?php echo $en('cal_outline'); ?>> <?php _e('Calendar Text Outline', 'free-short-term-rental-booking'); ?></label>
					</th>
					<td>
						<input type="color" name="vis_cal_outline" value="<?php echo esc_attr($visual['cal_outline'] ?? '#000000'); ?>">
						<input type="number" name="vis_cal_outline_size" value="<?php echo (int)($visual['cal_outline_size'] ?? 1); ?>" class="small-text"> px
						<span class="description"><?php _e('Creates a slight shadow/outline for readability', 'free-short-term-rental-booking'); ?></span>
					</td>
				</tr>
				<tr>
					<th scope="row">
						<label><input type="checkbox" name="vis_bs_color_en" class="fstrb-style-en" <?php echo $en('bs_color'); ?>> <?php _e('Calendar Text Shadow', 'free-short-term-rental-booking'); ?></label>
					</th>
					<td>
						<div style="display: flex; gap: 10px; align-items: center; flex-wrap: wrap;">
							<label title="<?php _e('Color', 'free-short-term-rental-booking'); ?>"><input type="color" name="vis_bs_color" value="<?php echo esc_attr($visual['bs_color'] ?? '#00000033'); ?>"></label>
							<label title="<?php _e('X Offset', 'free-short-term-rental-booking'); ?>">X: <input type="number" name="vis_bs_x" value="<?php echo (int)($visual['bs_x'] ?? 0); ?>" class="small-text"> px</label>
							<label title="<?php _e('Y Offset', 'free-short-term-rental-booking'); ?>">Y: <input type="number" name="vis_bs_y" value="<?php echo (int)($visual['bs_y'] ?? 4); ?>" class="small-text"> px</label>
							<label title="<?php _e('Blur', 'free-short-term-rental-booking'); ?>">Blur: <input type="number" name="vis_bs_blur" value="<?php echo (int)($visual['bs_blur'] ?? 10); ?>" class="small-text"> px</label>
							<label title="<?php _e('Spread', 'free-short-term-rental-booking'); ?>"><?php _e('Spread (Note: Text shadow might not support spread in all browsers)', 'free-short-term-rental-booking'); ?>: <input type="number" name="vis_bs_spread" value="<?php echo (int)($visual['bs_spread'] ?? 0); ?>" class="small-text"> px</label>
						</div>
						<p class="description"><?php _e('Applies a shadow to the numbers (dates) inside the calendar.', 'free-short-term-rental-booking'); ?></p>
					</td>
				</tr>
				<tr>
					<th scope="row">
						<?php _e('Calendar Dates Font', 'free-short-term-rental-booking'); ?>
					</th>
					<td>
						<div style="display: flex; gap: 15px; align-items: center; flex-wrap: wrap;">
							<label title="<?php _e('Font Size', 'free-short-term-rental-booking'); ?>"><input type="checkbox" name="vis_cal_date_size_en" class="fstrb-style-en" <?php echo $en('cal_date_size'); ?>> <input type="number" name="vis_cal_date_size" value="<?php echo (int)($visual['cal_date_size'] ?? 14); ?>" class="small-text"> px</label>
							<label title="<?php _e('Color', 'free-short-term-rental-booking'); ?>"><input type="checkbox" name="vis_cal_date_color_en" class="fstrb-style-en" <?php echo $en('cal_date_color'); ?>> <input type="color" name="vis_cal_date_color" value="<?php echo esc_attr($visual['cal_date_color'] ?? '#333333'); ?>"></label>
							<label title="<?php _e('Bold', 'free-short-term-rental-booking'); ?>"><input type="checkbox" name="vis_cal_date_weight_en" class="fstrb-style-en" <?php echo $en('cal_date_weight'); ?>> <?php _e('Bold', 'free-short-term-rental-booking'); ?> <input type="hidden" name="vis_cal_date_weight" value="bold"></label>
						</div>
						<p class="description"><?php _e('Styling for the day numbers in the calendar grid.', 'free-short-term-rental-booking'); ?></p>
					</td>
				</tr>
				<tr>
					<th scope="row">
						<?php _e('Calendar Labels Font', 'free-short-term-rental-booking'); ?>
					</th>
					<td>
						<div style="display: flex; gap: 15px; align-items: center; flex-wrap: wrap;">
							<label title="<?php _e('Font Size', 'free-short-term-rental-booking'); ?>"><input type="checkbox" name="vis_cal_lbl_size_en" class="fstrb-style-en" <?php echo $en('cal_lbl_size'); ?>> <input type="number" name="vis_cal_lbl_size" value="<?php echo (int)($visual['cal_lbl_size'] ?? 12); ?>" class="small-text"> px</label>
							<label title="<?php _e('Color', 'free-short-term-rental-booking'); ?>"><input type="checkbox" name="vis_cal_lbl_color_en" class="fstrb-style-en" <?php echo $en('cal_lbl_color'); ?>> <input type="color" name="vis_cal_lbl_color" value="<?php echo esc_attr($visual['cal_lbl_color'] ?? '#666666'); ?>"></label>
							<label title="<?php _e('Bold', 'free-short-term-rental-booking'); ?>"><input type="checkbox" name="vis_cal_lbl_weight_en" class="fstrb-style-en" <?php echo $en('cal_lbl_weight'); ?>> <?php _e('Bold', 'free-short-term-rental-booking'); ?> <input type="hidden" name="vis_cal_lbl_weight" value="bold"></label>
						</div>
						<p class="description"><?php _e('Styling for the "Available", "Occupied", etc. labels below the calendar.', 'free-short-term-rental-booking'); ?></p>
					</td>
				</tr>
			</table>

			<h3><?php _e('Services & Metadata', 'free-short-term-rental-booking'); ?></h3>
			<table class="form-table fstrb-styling-table">
				<tr>
					<th scope="row">
						<label><input type="checkbox" name="vis_svc_label_en" class="fstrb-style-en" <?php echo $en('svc_label'); ?>> <?php _e('Services Header Label', 'free-short-term-rental-booking'); ?></label>
					</th>
					<td><input type="text" name="vis_svc_label" value="<?php echo esc_attr($v_svc_label); ?>" class="regular-text" placeholder="<?php _e('Optional Services', 'free-short-term-rental-booking'); ?>"></td>
				</tr>
				<tr>
					<th scope="row">
						<label><input type="checkbox" name="vis_meta_style_en" class="fstrb-style-en" <?php echo $en('meta_style'); ?>> <?php _e('Metadata Style', 'free-short-term-rental-booking'); ?></label>
					</th>
					<td>
						<select name="vis_meta_style">
							<option value="list" <?php selected($v_meta_style, 'list'); ?>><?php _e('List (Default)', 'free-short-term-rental-booking'); ?></option>
							<option value="inline_bold" <?php selected($v_meta_style, 'inline_bold'); ?>><?php _e('Inline (Key Bold)', 'free-short-term-rental-booking'); ?></option>
							<option value="inline_normal" <?php selected($v_meta_style, 'inline_normal'); ?>><?php _e('Inline (Normal)', 'free-short-term-rental-booking'); ?></option>
						</select>
					</td>
				</tr>
				<tr>
					<th scope="row">
						<label><input type="checkbox" name="vis_meta_sep_en" class="fstrb-style-en" <?php echo $en('meta_sep'); ?>> <?php _e('Metadata Separator', 'free-short-term-rental-booking'); ?></label>
					</th>
					<td>
						<input type="text" name="vis_meta_sep" value="<?php echo esc_attr($v_meta_sep); ?>" class="small-text" placeholder="|">
						<span class="description"><?php _e('Symbol to separate inline metadata items', 'free-short-term-rental-booking'); ?></span>
					</td>
				</tr>
			</table>

			<h3><?php _e('Section Headings', 'free-short-term-rental-booking'); ?></h3>
			<table class="form-table fstrb-styling-table">
				<tr>
					<th scope="row">
						<label><input type="checkbox" name="vis_cal_h_text_en" class="fstrb-style-en" <?php echo $en('cal_h_text'); ?>> <?php _e('Calendar Heading', 'free-short-term-rental-booking'); ?></label>
					</th>
					<td>
						<input type="text" name="vis_cal_h_text" value="<?php echo esc_attr($visual['cal_h_text'] ?? __('Availability Calendar', 'free-short-term-rental-booking')); ?>" class="regular-text">
						<div class="fstrb-h-styling" style="margin-top: 10px; display: flex; gap: 15px; align-items: center; flex-wrap: wrap;">
							<label title="<?php _e('Font Size', 'free-short-term-rental-booking'); ?>"><input type="checkbox" name="vis_cal_h_size_en" class="fstrb-style-en" <?php echo $en('cal_h_size'); ?>> <input type="number" name="vis_cal_h_size" value="<?php echo (int)($visual['cal_h_size'] ?? 24); ?>" class="small-text"> px</label>
							<label title="<?php _e('Color', 'free-short-term-rental-booking'); ?>"><input type="checkbox" name="vis_cal_h_color_en" class="fstrb-style-en" <?php echo $en('cal_h_color'); ?>> <input type="color" name="vis_cal_h_color" value="<?php echo esc_attr($visual['cal_h_color'] ?? '#333333'); ?>"></label>
							<label title="<?php _e('Bold', 'free-short-term-rental-booking'); ?>"><input type="checkbox" name="vis_cal_h_weight_en" class="fstrb-style-en" <?php echo $en('cal_h_weight'); ?>> <?php _e('Bold', 'free-short-term-rental-booking'); ?> <input type="hidden" name="vis_cal_h_weight" value="bold"></label>
							<label title="<?php _e('Alignment', 'free-short-term-rental-booking'); ?>">
								<input type="checkbox" name="vis_cal_h_align_en" class="fstrb-style-en" <?php echo $en('cal_h_align'); ?>> 
								<select name="vis_cal_h_align">
									<option value="left" <?php selected($visual['cal_h_align'] ?? 'left', 'left'); ?>><?php _e('Left', 'free-short-term-rental-booking'); ?></option>
									<option value="center" <?php selected($visual['cal_h_align'] ?? 'left', 'center'); ?>><?php _e('Center', 'free-short-term-rental-booking'); ?></option>
									<option value="right" <?php selected($visual['cal_h_align'] ?? 'left', 'right'); ?>><?php _e('Right', 'free-short-term-rental-booking'); ?></option>
								</select>
							</label>
						</div>
					</td>
				</tr>
			</table>

			<h3><?php _e('Booking Button', 'free-short-term-rental-booking'); ?></h3>
			<table class="form-table fstrb-styling-table">
				<tr>
					<th scope="row">
						<label><input type="checkbox" name="vis_btn_text_en" class="fstrb-style-en" <?php echo $en('btn_text'); ?>> <?php _e('Button Label', 'free-short-term-rental-booking'); ?></label>
					</th>
					<td>
						<input type="text" name="vis_btn_text" value="<?php echo esc_attr($visual['btn_text'] ?? __('Send Booking', 'free-short-term-rental-booking')); ?>" class="regular-text" placeholder="<?php _e('Send Booking', 'free-short-term-rental-booking'); ?>">
					</td>
				</tr>
				<tr>
					<th scope="row">
						<?php _e('Typography', 'free-short-term-rental-booking'); ?>
					</th>
					<td>
						<div style="display: flex; gap: 15px; align-items: center; flex-wrap: wrap;">
							<label title="<?php _e('Font Family', 'free-short-term-rental-booking'); ?>">
								<input type="checkbox" name="vis_btn_font_en" class="fstrb-style-en" <?php echo $en('btn_font'); ?>> 
								<input type="text" name="vis_btn_font" value="<?php echo esc_attr($visual['btn_font'] ?? ''); ?>" placeholder="e.g. Arial, sans-serif" style="width: 150px;">
							</label>
							<label title="<?php _e('Font Size', 'free-short-term-rental-booking'); ?>">
								<input type="checkbox" name="vis_btn_size_en" class="fstrb-style-en" <?php echo $en('btn_size'); ?>> 
								<input type="number" name="vis_btn_size" value="<?php echo (int)($visual['btn_size'] ?? 16); ?>" class="small-text"> px
							</label>
							<label title="<?php _e('Bold', 'free-short-term-rental-booking'); ?>">
								<input type="checkbox" name="vis_btn_weight_en" class="fstrb-style-en" <?php echo $en('btn_weight'); ?>> <?php _e('Bold', 'free-short-term-rental-booking'); ?> 
								<input type="hidden" name="vis_btn_weight" value="bold">
							</label>
						</div>
					</td>
				</tr>
				<tr>
					<th scope="row">
						<?php _e('Visuals', 'free-short-term-rental-booking'); ?>
					</th>
					<td>
						<div style="display: flex; gap: 15px; align-items: center; flex-wrap: wrap;">
							<label title="<?php _e('Background Color', 'free-short-term-rental-booking'); ?>">
								<input type="checkbox" name="vis_btn_bg_en" class="fstrb-style-en" <?php echo $en('btn_bg'); ?>> 
								<input type="color" name="vis_btn_bg" value="<?php echo esc_attr($visual['btn_bg'] ?? '#007cba'); ?>">
							</label>
							<label title="<?php _e('Text Color', 'free-short-term-rental-booking'); ?>">
								<input type="checkbox" name="vis_btn_color_en" class="fstrb-style-en" <?php echo $en('btn_color'); ?>> 
								<input type="color" name="vis_btn_color" value="<?php echo esc_attr($visual['btn_color'] ?? '#ffffff'); ?>">
							</label>
							<label title="<?php _e('Border Radius', 'free-short-term-rental-booking'); ?>">
								<input type="checkbox" name="vis_btn_radius_en" class="fstrb-style-en" <?php echo $en('btn_radius'); ?>> 
								<input type="number" name="vis_btn_radius" value="<?php echo (int)($visual['btn_radius'] ?? 4); ?>" class="small-text"> px
							</label>
						</div>
					</td>
				</tr>
			</table>

			<script>
			document.getElementById('fstrb_toggle_all_styles').addEventListener('change', function() {
				const checkboxes = document.querySelectorAll('.fstrb-style-en');
				checkboxes.forEach(cb => cb.checked = this.checked);
			});
			</script>

			<p class="submit">
				<button type="submit" name="fstrb_save_styling" class="button button-primary"><?php _e('Save Changes', 'free-short-term-rental-booking'); ?></button>
			</p>
		</form>
		</div>
		<?php
		self::render_donation_box();
	}

	public static function page_ical() {
		global $wpdb;
		$t = FSTRB_DB::tables();
		$unit_id = isset($_GET['unit_id']) ? (int)$_GET['unit_id'] : 0;

		echo '<div class="wrap"><h1>' . __('iCal Imports', 'free-short-term-rental-booking') . '</h1>';
		echo '<form method="get" style="margin:10px 0">';
		echo '<input type="hidden" name="page" value="fstrb-ical" />';
		self::unit_select($unit_id);
		submit_button(__('Show', 'free-short-term-rental-booking'), 'secondary', '', false);
		echo '</form>';

		if (!$unit_id) { echo '</div>'; return; }

		// Delete action
		if (isset($_GET['action']) && $_GET['action'] === 'delete' && isset($_GET['id'])) {
			check_admin_referer('fstrb_delete_ical');
			$id = (int)$_GET['id'];
			$wpdb->delete($t['ical_feeds'], ['id' => $id, 'unit_id' => $unit_id]);
			echo '<div class="notice notice-success"><p>' . __('iCal feed deleted.', 'free-short-term-rental-booking') . '</p></div>';
		}

		if (isset($_POST['fstrb_add_feed'])) {
			check_admin_referer('fstrb_ical');
			$url = esc_url_raw($_POST['url'] ?? '');
			$provider = FSTRB_iCal::detect_provider($url);

			$wpdb->insert($t['ical_feeds'], [
				'unit_id' => $unit_id,
				'provider' => $provider,
				'url' => $url,
				'is_active' => !empty($_POST['is_active']) ? 1 : 0,
			]);
			echo '<div class="notice notice-success"><p>' . __('Feed saved.', 'free-short-term-rental-booking') . '</p></div>';
		}

		echo '<form method="post">';
		wp_nonce_field('fstrb_ical');
		echo '<h2>' . __('Add Import URL', 'free-short-term-rental-booking') . '</h2>';
		echo '<p>URL <input id="ical_url" name="url" style="width:600px" required placeholder="https://..."> ';
		echo '<label><input type="checkbox" name="is_active" checked> ' . __('active', 'free-short-term-rental-booking') . '</label> ';
		echo '<button class="button button-primary" name="fstrb_add_feed" value="1">' . __('Add', 'free-short-term-rental-booking') . '</button></p>';
		echo '</form>';

		$feeds = $wpdb->get_results($wpdb->prepare("SELECT * FROM {$t['ical_feeds']} WHERE unit_id=%d ORDER BY id DESC", $unit_id), ARRAY_A);
		echo '<h2>' . __('Imports', 'free-short-term-rental-booking') . '</h2><table class="widefat"><thead><tr><th>ID</th><th>' . __('Provider', 'free-short-term-rental-booking') . '</th><th>URL</th><th>' . __('Active', 'free-short-term-rental-booking') . '</th><th>' . __('Last Import', 'free-short-term-rental-booking') . '</th><th>' . __('Error', 'free-short-term-rental-booking') . '</th><th>' . __('Actions', 'free-short-term-rental-booking') . '</th></tr></thead><tbody>';
		
		$provider_labels = [
			'airbnb' => 'Airbnb',
			'booking' => 'Booking.com',
			'e_chalupy' => 'e-chalupy.cz',
			'megaubytovanie' => 'Megaubytovanie',
			'hauzi' => 'Hauzi',
			'bezrealitky' => 'BezRealitky',
			'other' => __('Other', 'free-short-term-rental-booking'),
		];

		foreach ($feeds as $f) {
			$del_url = wp_nonce_url("admin.php?page=fstrb-ical&unit_id=".$unit_id."&action=delete&id=".$f['id'], 'fstrb_delete_ical');
			// Format provider label nicely
			if (!isset($provider_labels[$f['provider']])) {
				// If it looks like a domain (contains a dot), display as-is
				// Otherwise capitalize it
				$prov_label = strpos($f['provider'], '.') !== false ? $f['provider'] : ucfirst($f['provider']);
			} else {
				$prov_label = $provider_labels[$f['provider']];
			}
			echo '<tr><td>'.esc_html($f['id']).'</td><td style="font-weight:bold">'.esc_html($prov_label).'</td><td><code>'.esc_html($f['url']).'</code></td><td>'.($f['is_active']?__('yes', 'free-short-term-rental-booking'):__('no', 'free-short-term-rental-booking')).'</td><td>'.esc_html($f['last_import_at']).'</td><td>'.esc_html(wp_trim_words($f['last_error'], 12)).'</td>';
			echo '<td><a href="'.$del_url.'" class="button button-small button-link-delete" onclick="return confirm(\''.esc_js(__('Delete this iCal feed?', 'free-short-term-rental-booking')).'\')" style="color:#a00">' . __('Delete', 'free-short-term-rental-booking') . '</a></td></tr>';
		}
		echo '</tbody></table>';

		echo '<p><a class="button" href="'.esc_url(admin_url('admin.php?page=fstrb-ical&unit_id='.$unit_id.'&do_import=1')).'">' . __('Run Import Now', 'free-short-term-rental-booking') . '</a></p>';

		if (!empty($_GET['do_import'])) {
			$res = FSTRB_iCal::import_unit_feeds($unit_id);
			echo '<div class="notice notice-info"><p>' . __('Import:', 'free-short-term-rental-booking') . ' ' . esc_html($res['message']) . '</p></div>';
		}

		echo '</div>';
		self::render_donation_box();
	}

	public static function page_contact_fields() {
		global $wpdb;
		$t = FSTRB_DB::tables();

		echo '<div class="wrap"><h1>' . __('Contact Fields', 'free-short-term-rental-booking') . '</h1>';

		// Delete action
		if (isset($_GET['action']) && $_GET['action'] === 'delete' && isset($_GET['id'])) {
			check_admin_referer('fstrb_delete_field');
			$id = (int)$_GET['id'];
			$wpdb->delete($t['contact_fields'], ['id' => $id]);
			$wpdb->delete($t['contact_field_meta'], ['field_id' => $id]);
			echo '<div class="notice notice-success"><p>' . __('Field deleted.', 'free-short-term-rental-booking') . '</p></div>';
		}

		// Edit/Add logic
		$edit_id = isset($_REQUEST['edit_id']) ? (int)$_REQUEST['edit_id'] : 0;
		$edit_data = null;
		if ($edit_id) {
			$edit_data = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$t['contact_fields']} WHERE id=%d", $edit_id));
		}

		// Save field
		if (isset($_POST['fstrb_save_field'])) {
			check_admin_referer('fstrb_contact_fields');
			$data = [
				'field_key' => strtolower(preg_replace('/[^a-z0-9_]/', '', sanitize_text_field($_POST['field_key'] ?? ''))),
				'is_active' => !empty($_POST['is_active']) ? 1 : 0,
				'sort_order' => (int)($_POST['sort_order'] ?? 0),
			];

			$field_id = 0;
			if ($edit_id && $edit_data) {
				$wpdb->update($t['contact_fields'], $data, ['id'=>$edit_id]);
				echo '<div class="notice notice-success"><p>' . __('Updated.', 'free-short-term-rental-booking') . '</p></div>';
				$field_id = $edit_id;
			} else {
				$wpdb->insert($t['contact_fields'], $data);
				$field_id = $wpdb->insert_id;
				echo '<div class="notice notice-success"><p>' . __('Saved.', 'free-short-term-rental-booking') . '</p></div>';
			}

			// Save Meta (Labels)
			if ($field_id && isset($_POST['meta'])) {
				$wpdb->delete($t['contact_field_meta'], ['field_id' => $field_id]);
				foreach ($_POST['meta'] as $pair) {
					if (empty($pair['key'])) continue;
					$wpdb->insert($t['contact_field_meta'], [
						'field_id' => $field_id,
						'meta_key' => sanitize_text_field($pair['key']),
						'meta_value' => sanitize_textarea_field($pair['value']),
						'locale' => !empty($pair['locale']) ? sanitize_text_field($pair['locale']) : null,
					]);
				}
			}
			$edit_id = 0; $edit_data = null; // reset
		}

		// Form
		$lbl = $edit_id ? __('Edit Field', 'free-short-term-rental-booking') : __('Add Field', 'free-short-term-rental-booking');
		$btn = $edit_id ? __('Update', 'free-short-term-rental-booking') : __('Add', 'free-short-term-rental-booking');
		$val_key = $edit_data ? $edit_data->field_key : '';
		$val_order = $edit_data ? $edit_data->sort_order : '0';
		$val_active = (!$edit_data || $edit_data->is_active) ? 'checked' : '';

		echo '<form method="post" style="background:#fff;padding:15px;border:1px solid #ccd0d4;margin-bottom:20px">';
		wp_nonce_field('fstrb_contact_fields');
		echo '<input type="hidden" name="edit_id" value="'.$edit_id.'">';
		echo '<h3>' . $lbl . ' ' . ($edit_id ? '<a href="admin.php?page=fstrb-contacts" class="page-title-action">'.__('Cancel', 'free-short-term-rental-booking').'</a>' : '') . '</h3>';
		echo '<div style="display:flex;gap:15px;align-items:flex-end;flex-wrap:wrap">';
		echo '<div><label>' . __('Unique Key', 'free-short-term-rental-booking') . ' (e.g. flight_no)<br><input name="field_key" required value="'.esc_attr($val_key).'"></label></div>';
		echo '<div><label>' . __('Sort Order', 'free-short-term-rental-booking') . '<br><input type="number" name="sort_order" value="'.esc_attr($val_order).'" style="width:70px"></label></div>';
		echo '<div><label><input type="checkbox" name="is_active" '.$val_active.'> ' . __('active', 'free-short-term-rental-booking') . '</label></div>';
		echo '<div><button class="button button-primary" name="fstrb_save_field" value="1">' . $btn . '</button></div>';
		echo '</div>';

		// Labels (Metadata)
		$meta = [];
		if ($edit_id) {
			$meta = $wpdb->get_results($wpdb->prepare("SELECT * FROM {$t['contact_field_meta']} WHERE field_id=%d", $edit_id), ARRAY_A);
		} else {
			// Default "label" keys for common locales if adding new
			$meta = [['meta_key'=>'label', 'meta_value'=>'', 'locale'=>'']];
		}

		$locales = ['' => __('All (Global)', 'free-short-term-rental-booking'), 'cs' => 'CZ', 'en' => 'EN', 'de' => 'DE', 'pl' => 'PL'];
		?>
		<div style="margin-top:15px;border-top:1px solid #eee;padding-top:15px">
			<strong><?php _e('Labels (per Language)', 'free-short-term-rental-booking'); ?></strong>
			<div id="fstrb_labels_wrapper" style="margin-top:10px">
				<?php foreach ($meta as $idx => $m): ?>
				<div class="fstrb-meta-row" style="margin-bottom:5px;display:flex;gap:5px">
					<select name="meta[<?php echo $idx; ?>][locale]" style="width:80px">
						<?php foreach ($locales as $k=>$v) echo '<option value="'.$k.'" '.selected($m['locale'], $k, false).'>'.$v.'</option>'; ?>
					</select>
					<input type="hidden" name="meta[<?php echo $idx; ?>][key]" value="label">
					<input type="text" name="meta[<?php echo $idx; ?>][value]" placeholder="<?php _e('Label text...', 'free-short-term-rental-booking'); ?>" value="<?php echo esc_attr($m['meta_value']); ?>" size="40">
					<button type="button" class="button button-small" onclick="this.parentNode.remove()">x</button>
				</div>
				<?php endforeach; ?>
			</div>
			<button type="button" class="button button-small" onclick="addLabelRow()"><?php _e('+ Add Label', 'free-short-term-rental-booking'); ?></button>
			<script>
			function addLabelRow() {
				const w = document.getElementById('fstrb_labels_wrapper');
				const idx = w.children.length + new Date().getTime();
				const div = document.createElement('div');
				div.className = 'fstrb-meta-row';
				div.style = 'margin-bottom:5px;display:flex;gap:5px';
				div.innerHTML = `<select name="meta[${idx}][locale]" style="width:80px">
					<option value=""><?php _e('All', 'free-short-term-rental-booking'); ?></option>
					<option value="cs">CZ</option>
					<option value="en">EN</option>
					<option value="de">DE</option>
				</select> <input type="hidden" name="meta[${idx}][key]" value="label"> <input type="text" name="meta[${idx}][value]" placeholder="<?php _e('Label text...', 'free-short-term-rental-booking'); ?>" size="40"> <button type="button" class="button button-small" onclick="this.parentNode.remove()">x</button>`;
				w.appendChild(div);
			}
			</script>
		</div>
		</form>
		<?php

		// List
		$fields = $wpdb->get_results("SELECT * FROM {$t['contact_fields']} ORDER BY sort_order ASC, id DESC", ARRAY_A);
		echo '<h2>' . __('Available Contact Fields', 'free-short-term-rental-booking') . '</h2>';
		echo '<table class="widefat striped"><thead><tr><th>ID</th><th>' . __('Key', 'free-short-term-rental-booking') . '</th><th>' . __('Labels', 'free-short-term-rental-booking') . '</th><th>' . __('Order', 'free-short-term-rental-booking') . '</th><th>' . __('Active', 'free-short-term-rental-booking') . '</th><th>' . __('Actions', 'free-short-term-rental-booking') . '</th></tr></thead><tbody>';
		foreach ($fields as $f) {
			$labels = $wpdb->get_results($wpdb->prepare("SELECT locale, meta_value FROM {$t['contact_field_meta']} WHERE field_id=%d AND meta_key='label'", $f['id']), ARRAY_A);
			$label_str = [];
			foreach ($labels as $l) {
				$loc = $l['locale'] ?: 'global';
				$label_str[] = "<strong>$loc:</strong> " . esc_html($l['meta_value']);
			}

			$del_url = wp_nonce_url("admin.php?page=fstrb-contacts&action=delete&id=".$f['id'], 'fstrb_delete_field');
			$edit_url = "admin.php?page=fstrb-contacts&edit_id=".$f['id'];
			echo '<tr>
				<td>'.esc_html($f['id']).'</td>
				<td><code>'.esc_html($f['field_key']).'</code></td>
				<td>'.implode(' | ', $label_str).'</td>
				<td>'.esc_html($f['sort_order']).'</td>
				<td>'.($f['is_active']?__('yes', 'free-short-term-rental-booking'):__('no', 'free-short-term-rental-booking')).'</td>
				<td>
					<a href="'.$edit_url.'" class="button button-small">'.__('Edit', 'free-short-term-rental-booking').'</a>
					<a href="'.$del_url.'" class="button button-small button-link-delete" onclick="return confirm(\''.esc_js(__('Are you sure?', 'free-short-term-rental-booking')).'\')">'.__('Delete', 'free-short-term-rental-booking').'</a>
				</td>
			</tr>';
		}
		if (!$fields) echo '<tr><td colspan="6">'.__('No custom fields defined.', 'free-short-term-rental-booking').'</td></tr>';
		echo '</tbody></table></div>';
		self::render_donation_box();
	}

	private static function render_donation_box() {
		?>
		<div style="margin-top: 40px; padding: 20px; background: #fff; border: 1px solid #ccd0d4; border-radius: 4px; max-width: 800px; display: flex; align-items: center; gap: 20px; box-shadow: 0 1px 1px rgba(0,0,0,.04);">
			<div style="font-size: 40px; color: #f0c330;">
				<span class="dashicons dashicons-coffee" style="font-size: inherit; width: auto; height: auto;"></span>
			</div>
			<div style="flex-grow: 1;">
				<h3 style="margin-top: 0; margin-bottom: 5px;"><?php _e('Support Developer', 'free-short-term-rental-booking'); ?></h3>
				<p style="margin: 0; color: #666;"><?php _e('If you like this plugin and find it useful, please consider a small donation to support further development and maintenance. Help me keep it free!', 'free-short-term-rental-booking'); ?></p>
			</div>
			<div>
				<a href="https://buymeacoffee.com/vachr" target="_blank" class="button button-primary button-hero" style="background: #FFDD00; border-color: #FFDD00; color: #000000; display: flex; align-items: center; gap: 8px; font-weight: bold;">
					<span class="dashicons dashicons-coffee" style="margin-top: 4px;"></span>
					<?php _e('Buy me a coffee', 'free-short-term-rental-booking'); ?>
				</a>
			</div>
		</div>
		<?php
	}
}

add_action('admin_menu', ['FSTRB_Admin', 'register']);
