<?php
if (!defined('ABSPATH')) exit;

class FSTRB_CPT {
	public static function get_service_pricing_types() {
		return [
			'per_stay' => __('per stay', 'free-short-term-rental-booking'),
			'per_night' => __('per night', 'free-short-term-rental-booking'),
			'per_person' => __('per person', 'free-short-term-rental-booking'),
			'per_person_night' => __('per person/night', 'free-short-term-rental-booking'),
			'free' => __('free (service)', 'free-short-term-rental-booking'),
		];
	}

	public static function register() {
		register_post_type('fstrb_unit', [
			'labels' => [
				'name' => __('Apartments', 'free-short-term-rental-booking'),
				'singular_name' => __('Apartment', 'free-short-term-rental-booking'),
				'add_new_item' => __('Add Apartment', 'free-short-term-rental-booking'),
				'edit_item' => __('Edit Apartment', 'free-short-term-rental-booking'),
			],
			'public' => false,
			'show_ui' => true,
			'menu_icon' => 'dashicons-building',
			'supports' => ['title','editor','thumbnail'],
			'has_archive' => false,
			'show_in_rest' => true,
		]);

		// Taxonomy for amenities
		register_taxonomy('fstrb_amenity', ['fstrb_unit'], [
			'labels' => [
				'name' => __('Amenities', 'free-short-term-rental-booking'),
				'singular_name' => __('Amenity', 'free-short-term-rental-booking'),
			],
			'public' => false,
			'show_ui' => true,
			'hierarchical' => false,
			'show_in_rest' => true,
		]);

		add_action('add_meta_boxes', [__CLASS__, 'metaboxes']);
		add_action('save_post_fstrb_unit', [__CLASS__, 'save_meta'], 10, 2);

		// Custom columns
		add_filter('manage_fstrb_unit_posts_columns', [__CLASS__, 'columns_head']);
		add_action('manage_fstrb_unit_posts_custom_column', [__CLASS__, 'columns_content'], 10, 2);
		add_action('admin_footer', [__CLASS__, 'render_table_footer']);
	}

	public static function columns_head($columns) {
		// Add new column after title
		$new = [];
		foreach($columns as $key => $title) {
			$new[$key] = $title;
			if ($key === 'title') {
				$new['shortcodes'] = __('Shortcodes', 'free-short-term-rental-booking');
			}
		}
		return $new;
	}

	public static function columns_content($column_name, $post_id) {
		if ($column_name == 'shortcodes') {
			echo '<div style="display:flex;align-items:center;gap:10px">';
			echo '<code style="font-size:11px;color:#0073aa;user-select:all;margin:0">[fstrb_booking_form unit="'.$post_id.'"]</code>';
			echo '<a href="#" style="font-size:10px;text-decoration:none;color:#555;white-space:nowrap" onclick="var s=this.parentNode.nextElementSibling;s.style.display=s.style.display==\'none\'?\'block\':\'none\';return false">'.__('Show more', 'free-short-term-rental-booking').' &#9662;</a>';
			echo '</div>';
			
			// Hidden other shortcodes
			echo '<div style="display:none;margin-top:5px;padding:5px;background:#f0f0f1;border:1px solid #ddd">';
			echo '<code style="display:block;margin-bottom:2px;font-size:10px;user-select:all">[fstrb_availability_calendar unit="'.$post_id.'"]</code>';
			echo '</div>';
		}
	}

	public static function render_table_footer() {
		$screen = get_current_screen();
		if (!$screen || $screen->post_type !== 'fstrb_unit' || $screen->base !== 'edit') return;
		
		// Output as hidden template then move with JS to proper location
		?>
		<div id="fstrb-shortcode-help" style="display:none;margin-top:20px;color:#444;font-size:13px;background:#fff;padding:20px;border:1px solid #ccd0d4;border-left:4px solid #0073aa;box-shadow:0 1px 1px rgba(0,0,0,.04);max-width:1200px">
			<h3 style="margin:0 0 10px 0;font-size:14px;border-bottom:1px solid #eee;padding-bottom:5px"><?php _e('Shortcode Guide & Examples', 'free-short-term-rental-booking'); ?></h3>
			<p style="margin-bottom:5px"><?php _e('You can customize the calendar/form using these parameters in the shortcode:', 'free-short-term-rental-booking'); ?></p>
			<ul style="list-style:disc;margin-left:20px;margin-bottom:15px">
				<li><code>months="NUMBER"</code> – <?php _e('Number of months to display (e.g. 1, 3, 6, 12). Default is 1.', 'free-short-term-rental-booking'); ?></li>
				<li><code>grid="NUMBER"</code> – <?php _e('How many months to show per row (grid layout). Useful for multi-month views.', 'free-short-term-rental-booking'); ?></li>
				<li><code>currency="CODE"</code> – <?php _e('Currency to use for prices (e.g. EUR, USD, CZK). Must be one of the defined active currencies.', 'free-short-term-rental-booking'); ?></li>
			</ul>
			<strong><?php _e('Examples:', 'free-short-term-rental-booking'); ?></strong><br>
			<div style="background:#f0f0f1;padding:10px;margin-top:5px;border:1px solid #ddd">
				<code>[fstrb_availability_calendar unit="123" months="3" grid="3"]</code> <span style="color:#666"> – <?php _e('Shows a 3-month availability calendar in a single row.', 'free-short-term-rental-booking'); ?></span><br>
				<code>[fstrb_availability_calendar unit="123" months="12" grid="4"]</code> <span style="color:#666"> – <?php _e('Shows a full year calendar, 4 months per row.', 'free-short-term-rental-booking'); ?></span><br>
				<code>[fstrb_booking_form unit="123" currency="USD"]</code> <span style="color:#666"> – <?php _e('Shows the booking form with prices in USD.', 'free-short-term-rental-booking'); ?></span>
			</div>
		</div>
		<script>
		document.addEventListener('DOMContentLoaded', function() {
			var helpInfo = document.getElementById('fstrb-shortcode-help');
			var wrapper = document.querySelector('.wrap');
			if (helpInfo && wrapper) {
				wrapper.appendChild(helpInfo);
				helpInfo.style.display = 'block';
			}
		});
		</script>
		<?php
	}

	public static function metaboxes() {
		add_meta_box('fstrb_unit_details', __('Apartment Details', 'free-short-term-rental-booking'), [__CLASS__, 'render_details'], 'fstrb_unit', 'normal', 'high');
		add_meta_box('fstrb_unit_pricing', __('Pricing (Base)', 'free-short-term-rental-booking'), [__CLASS__, 'render_pricing'], 'fstrb_unit', 'normal', 'default');
		add_meta_box('fstrb_unit_services', __('Services', 'free-short-term-rental-booking'), [__CLASS__, 'render_services'], 'fstrb_unit', 'normal', 'default');
		add_meta_box('fstrb_unit_contacts', __('Contact Fields', 'free-short-term-rental-booking'), [__CLASS__, 'render_contacts'], 'fstrb_unit', 'normal', 'default');
		add_meta_box('fstrb_unit_ical', __('iCal Sync', 'free-short-term-rental-booking'), [__CLASS__, 'render_ical'], 'fstrb_unit', 'side', 'default');
	}

	private static function field($name, $value, $type='text', $placeholder='') {
		$name_attr = esc_attr($name);
		$val = esc_attr($value);
		$ph = esc_attr($placeholder);
		return "<input style=\"width:100%\" type=\"$type\" name=\"$name_attr\" value=\"$val\" placeholder=\"$ph\" />";
	}

	public static function render_details($post) {
		wp_nonce_field('fstrb_unit_meta', 'fstrb_unit_meta_nonce');
		$checkin  = get_post_meta($post->ID, '_fstrb_checkin', true);
		$checkout = get_post_meta($post->ID, '_fstrb_checkout', true);
		$instructions = get_post_meta($post->ID, '_fstrb_instructions', true);
		$capacity = get_post_meta($post->ID, '_fstrb_capacity', true);
		$min_stay = get_post_meta($post->ID, '_fstrb_min_stay', true);
		$max_stay = get_post_meta($post->ID, '_fstrb_max_stay', true);

		$is_paused = get_post_meta($post->ID, '_fstrb_is_paused', true);

		echo '<p><label><strong>'.esc_html__('Pause Apartment', 'free-short-term-rental-booking').'</strong></label><br/>'.
			 '<label><input type="checkbox" name="_fstrb_is_paused" value="1" '.checked($is_paused, 1, false).' /> '.esc_html__('Temporarily stop new bookings', 'free-short-term-rental-booking').'</label></p>';

		echo '<p><label><strong>'.esc_html__('Check-in time', 'free-short-term-rental-booking').'</strong></label><br/>'.
			 self::field('_fstrb_checkin', $checkin, 'text', 'e.g. 15:00').'</p>';
		echo '<p><label><strong>'.esc_html__('Check-out time', 'free-short-term-rental-booking').'</strong></label><br/>'.
			 self::field('_fstrb_checkout', $checkout, 'text', 'e.g. 10:00').'</p>';
		echo '<p><label><strong>'.esc_html__('Capacity (guests)', 'free-short-term-rental-booking').'</strong></label><br/>'.
			 self::field('_fstrb_capacity', $capacity, 'number', 'e.g. 4').'</p>';
		echo '<p><label><strong>'.esc_html__('Minimum stay (days)', 'free-short-term-rental-booking').'</strong></label><br/>'.
			 self::field('_fstrb_min_stay', $min_stay, 'number', 'e.g. 2').'</p>';
		echo '<p><label><strong>'.esc_html__('Maximum stay (days)', 'free-short-term-rental-booking').'</strong></label><br/>'.
			 self::field('_fstrb_max_stay', $max_stay, 'number', 'e.g. 30').'</p>';
		echo '<p><label><strong>'.esc_html__('Booking Instructions', 'free-short-term-rental-booking').'</strong></label><br/>'.
			 '<textarea style="width:100%;min-height:120px" name="_fstrb_instructions">'.esc_textarea($instructions).'</textarea></p>';
		echo '<p style="color:#666">'.esc_html__('Set amenities in the "Amenities" box on the right (tags).', 'free-short-term-rental-booking').'</p>';
	}

	public static function render_pricing($post) {
		$mode = get_post_meta($post->ID, '_fstrb_price_mode', true) ?: 'per_unit_night';
		$pppn = get_post_meta($post->ID, '_fstrb_price_per_person_night', true);
		$pun  = get_post_meta($post->ID, '_fstrb_price_per_unit_night', true);
		$per_stay = get_post_meta($post->ID, '_fstrb_price_per_stay', true);
		$per_night = get_post_meta($post->ID, '_fstrb_price_per_night', true);
		$per_person = get_post_meta($post->ID, '_fstrb_price_per_person', true);


		echo '<p><label><strong>'.esc_html__('Price mode', 'free-short-term-rental-booking').'</strong></label><br/>';
		echo '<select name="_fstrb_price_mode" style="width:100%">';
		$opts = [
			'per_unit_night' => __('Price per unit / night', 'free-short-term-rental-booking'),
			'per_person_night' => __('Price per person / night', 'free-short-term-rental-booking'),
			'per_stay' => __('Price per stay (one-time)', 'free-short-term-rental-booking'),
			'per_night' => __('Price per night', 'free-short-term-rental-booking'),
			'per_person' => __('Price per person (one-time)', 'free-short-term-rental-booking'),
		];
		foreach ($opts as $k=>$lab) {
			printf('<option value="%s" %s>%s</option>', esc_attr($k), selected($mode,$k,false), esc_html($lab));
		}
		echo '</select></p>';

		$currencies = array_map('trim', explode(',', get_option('fstrb_available_currencies', 'EUR')));
		foreach ($currencies as $c) {
			$c_pppn = get_post_meta($post->ID, '_fstrb_price_per_person_night_' . $c, true);
			$c_pun  = get_post_meta($post->ID, '_fstrb_price_per_unit_night_' . $c, true);
			$c_per_stay = get_post_meta($post->ID, '_fstrb_price_per_stay_' . $c, true);
			$c_per_night = get_post_meta($post->ID, '_fstrb_price_per_night_' . $c, true);
			$c_per_person = get_post_meta($post->ID, '_fstrb_price_per_person_' . $c, true);

			echo '<div style="background:#f9f9f9;padding:10px;margin-bottom:10px;border:1px solid #ddd;">';
			echo '<h4>' . sprintf(__('Prices in %s', 'free-short-term-rental-booking'), $c) . '</h4>';
			echo '<div style="display:grid;grid-template-columns:1fr 1fr;gap:10px">';
			echo '<p><label><strong>'.esc_html__('Price per person / night', 'free-short-term-rental-booking').'</strong></label><br/>'.
				 self::field('_fstrb_price_per_person_night_'.$c, $c_pppn, 'number', 'e.g. 35').'</p>';
			echo '<p><label><strong>'.esc_html__('Price per unit / night', 'free-short-term-rental-booking').'</strong></label><br/>'.
				 self::field('_fstrb_price_per_unit_night_'.$c, $c_pun, 'number', 'e.g. 120').'</p>';
			echo '<p><label><strong>'.esc_html__('Price per stay (one-time)', 'free-short-term-rental-booking').'</strong></label><br/>'.
				 self::field('_fstrb_price_per_stay_'.$c, $c_per_stay, 'number', 'e.g. 500').'</p>';
			echo '<p><label><strong>'.esc_html__('Price per night', 'free-short-term-rental-booking').'</strong></label><br/>'.
				 self::field('_fstrb_price_per_night_'.$c, $c_per_night, 'number', 'e.g. 100').'</p>';
			echo '<p><label><strong>'.esc_html__('Price per person (one-time)', 'free-short-term-rental-booking').'</strong></label><br/>'.
				 self::field('_fstrb_price_per_person_'.$c, $c_per_person, 'number', 'e.g. 50').'</p>';
			echo '</div></div>';
		}

		echo '<hr/><p style="color:#666">'.esc_html__('Seasonal prices (Christmas, New Year, weekend...) can be set in the "Apartments → Rates" menu.', 'free-short-term-rental-booking').'</p>';
	}

	public static function render_services($post) {
		global $wpdb;
		$t = FSTRB_DB::tables();

		$all_services = $wpdb->get_results("SELECT * FROM {$t['services']} WHERE is_active=1 ORDER BY sort_order ASC, name ASC");
		// Get dictionary of unit services keyed by service_id and currency
		$unit_services_raw = $wpdb->get_results($wpdb->prepare("SELECT * FROM {$t['unit_services']} WHERE unit_id=%d", $post->ID));
		$my_services = [];
		foreach ($unit_services_raw as $us) {
			$my_services[$us->service_id][$us->currency] = $us;
		}

		$currencies = array_map('trim', explode(',', get_option('fstrb_available_currencies', 'EUR')));

		echo '<p style="color:#666">'.__('Select services available for this unit. You can override price or make them mandatory.', 'free-short-term-rental-booking').'</p>';

		if (!$all_services) {
			echo '<p>'.__('No active services defined.', 'free-short-term-rental-booking').' <a href="admin.php?page=fstrb-services">'.__('Add Services', 'free-short-term-rental-booking').'</a></p>';
			return;
		}

		echo '<table class="widefat striped">';
		echo '<thead><tr>
			<th style="width:30px;"><input type="checkbox" onclick="jQuery(\'.fstrb-svc-cb\').prop(\'checked\',this.checked)"></th>
			<th>'.__('Service', 'free-short-term-rental-booking').'</th>
			<th>'.__('Currency', 'free-short-term-rental-booking').'</th>
			<th>'.__('Default Price', 'free-short-term-rental-booking').'</th>
			<th>'.__('Custom Price', 'free-short-term-rental-booking').'</th>
			<th>'.__('Pricing Type', 'free-short-term-rental-booking').'</th>
			<th>'.__('Mode', 'free-short-term-rental-booking').'</th>
		</tr></thead><tbody>';

		$pricing_options = self::get_service_pricing_types();

		foreach ($all_services as $s) {
			$row_span = count($currencies);
			$first_c = true;
			foreach ($currencies as $c) {
				$has = isset($my_services[$s->id][$c]);
				$us = $has ? $my_services[$s->id][$c] : null;
				$checked = $has ? 'checked' : '';
				
				// Custom Price
				$cust_price = ($has && !is_null($us->price_override)) ? (float)$us->price_override : '';
				
				// Custom Pricing Type
				$cust_pricing_type = ($has && !empty($us->pricing_type)) ? $us->pricing_type : '';

				// Optional/Mandatory
				$is_opt = ($has && isset($us->is_optional)) ? (int)$us->is_optional : (isset($s->is_optional) ? (int)$s->is_optional : 1); 

				// Default Price from service meta
				$def_price = $wpdb->get_var($wpdb->prepare("SELECT meta_value FROM {$t['service_meta']} WHERE service_id=%d AND meta_key=%s", $s->id, 'price_' . $c));

				echo '<tr>';
				if ($first_c) {
					echo '<td rowspan="'.$row_span.'"><input type="checkbox" class="fstrb-svc-cb" name="fstrb_services['.$s->id.'][enable]" value="1" '.(isset($my_services[$s->id])?'checked':'').'></td>';
					echo '<td rowspan="'.$row_span.'"><strong>'.esc_html($s->name).'</strong><br><small style="color:#888">'.esc_html($s->code).'</small></td>';
				}
				echo '<td><strong>'.esc_html($c).'</strong></td>';
				echo '<td>'.esc_html($def_price ?: 0).' <small>('.esc_html($pricing_options[$s->pricing_type] ?? $s->pricing_type).')</small></td>';
				echo '<td><input type="number" step="0.01" name="fstrb_services['.$s->id.']['.$c.'][price]" value="'.esc_attr($cust_price).'" placeholder="'.esc_attr($def_price ?: 0).'" style="width:80px"></td>';
				echo '<td>
					<select name="fstrb_services['.$s->id.']['.$c.'][pricing_type]">
						<option value="">-- ' . __('Default', 'free-short-term-rental-booking') . ' --</option>';
				foreach ($pricing_options as $pk => $pv) {
					printf('<option value="%s" %s>%s</option>', esc_attr($pk), selected($cust_pricing_type, $pk, false), esc_html($pv));
				}
				echo '</select>
				</td>';
				echo '<td>
					<select name="fstrb_services['.$s->id.']['.$c.'][optional]">
						<option value="1" '.selected($is_opt,1,false).'>'.__('Optional', 'free-short-term-rental-booking').'</option>
						<option value="0" '.selected($is_opt,0,false).'>'.__('Mandatory', 'free-short-term-rental-booking').'</option>
					</select>
				</td>';
				echo '</tr>';
				$first_c = false;
			}
		}
		echo '</tbody></table>';
	}

	public static function render_contacts($post) {
		global $wpdb;
		$t = FSTRB_DB::tables();

		$all_fields = $wpdb->get_results("SELECT id, field_key FROM {$t['contact_fields']} WHERE is_active=1 ORDER BY sort_order ASC");
		$my_fields = $wpdb->get_col($wpdb->prepare("SELECT field_id FROM {$t['unit_contact_fields']} WHERE unit_id=%d", $post->ID));

		echo '<p style="color:#666">'.__('Select custom contact fields to be displayed in the booking form for this apartment.', 'free-short-term-rental-booking').'</p>';

		if (!$all_fields) {
			echo '<p>'.__('No active contact fields defined.', 'free-short-term-rental-booking').' <a href="admin.php?page=fstrb-contacts">'.__('Add Fields', 'free-short-term-rental-booking').'</a></p>';
			return;
		}

		echo '<div style="display:grid;grid-template-columns:repeat(auto-fill,minmax(200px,1fr));gap:10px">';
		foreach ($all_fields as $f) {
			$checked = in_array($f->id, $my_fields) ? 'checked' : '';
			// Get a label for current user language to show what it is
			$label = $wpdb->get_var($wpdb->prepare("SELECT meta_value FROM {$t['contact_field_meta']} WHERE field_id=%d AND meta_key='label' AND locale=%s", $f->id, substr(get_user_locale(), 0, 2)));
			if (!$label) $label = $wpdb->get_var($wpdb->prepare("SELECT meta_value FROM {$t['contact_field_meta']} WHERE field_id=%d AND meta_key='label' AND (locale IS NULL OR locale='')", $f->id));
			if (!$label) $label = $f->field_key;

			printf('<label><input type="checkbox" name="fstrb_contact_fields[]" value="%d" %s> %s <small>(<code>%s</code>)</small></label>', $f->id, $checked, esc_html($label), esc_html($f->field_key));
		}
		echo '</div>';
	}

	public static function render_ical($post) {
		$token = get_post_meta($post->ID, '_fstrb_ical_token', true);
		if (!$token) {
			$token = wp_generate_password(20, false, false);
			update_post_meta($post->ID, '_fstrb_ical_token', $token);
		}
		$export_url = rest_url("fstrb/v1/ical/export/{$post->ID}/{$token}.ics");

		echo '<p><strong>'.esc_html__('iCal Export URL', 'free-short-term-rental-booking').'</strong></p>';
		echo '<p><input style="width:100%" readonly value="'.esc_attr($export_url).'" /></p>';
		echo '<p style="color:#666">'.esc_html__('Paste this URL into Airbnb/Booking as a calendar import.', 'free-short-term-rental-booking').'</p>';
		echo '<p><a class="button button-secondary" href="'.esc_url(admin_url('admin.php?page=fstrb-ical&unit_id='.$post->ID)).'">'.esc_html__('Configure Imports', 'free-short-term-rental-booking').'</a></p>';
	}

	public static function save_meta($post_id, $post) {
		if (!isset($_POST['fstrb_unit_meta_nonce']) || !wp_verify_nonce($_POST['fstrb_unit_meta_nonce'], 'fstrb_unit_meta')) return;
		if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
		if (!current_user_can('edit_post', $post_id)) return;

		$fields = [
			'_fstrb_checkin','_fstrb_checkout','_fstrb_instructions','_fstrb_capacity',
			'_fstrb_price_mode',

			'_fstrb_min_stay', '_fstrb_max_stay'
		];
		$currencies = array_map('trim', explode(',', get_option('fstrb_available_currencies', 'EUR')));
		foreach ($currencies as $c) {
			$fields[] = '_fstrb_price_per_person_night_' . $c;
			$fields[] = '_fstrb_price_per_unit_night_' . $c;
			$fields[] = '_fstrb_price_per_stay_' . $c;
			$fields[] = '_fstrb_price_per_night_' . $c;
			$fields[] = '_fstrb_price_per_person_' . $c;
		}

		foreach ($fields as $f) {
			if (!isset($_POST[$f])) continue;
			$val = $_POST[$f];
			if ($f === '_fstrb_instructions') $val = wp_kses_post($val);
			else $val = sanitize_text_field($val);
			update_post_meta($post_id, $f, $val);
		}

		// Save Pause status
		$is_paused = !empty($_POST['_fstrb_is_paused']) ? 1 : 0;
		update_post_meta($post_id, '_fstrb_is_paused', $is_paused);

		// Save Services
		global $wpdb;
		$t = FSTRB_DB::tables();
		$wpdb->delete($t['unit_services'], ['unit_id' => $post_id]);
		
		if (isset($_POST['fstrb_services']) && is_array($_POST['fstrb_services'])) {
			foreach ($_POST['fstrb_services'] as $sid => $svc_data) {
				if (empty($svc_data['enable'])) continue;

				foreach ($currencies as $c) {
					$c_data = $svc_data[$c] ?? [];
					$price_override = (isset($c_data['price']) && $c_data['price'] !== '') ? (float)$c_data['price'] : null;
					$is_opt = isset($c_data['optional']) ? (int)$c_data['optional'] : 1;
					$pricing_type = !empty($c_data['pricing_type']) ? sanitize_text_field($c_data['pricing_type']) : null;
					
					$wpdb->insert($t['unit_services'], [
						'unit_id' => $post_id,
						'service_id' => (int)$sid,
						'currency' => $c,
						'is_optional' => $is_opt,
						'price_override' => $price_override,
						'pricing_type' => $pricing_type
					]);
				}
			}
		}

		// Save Contact Fields
		$wpdb->delete($t['unit_contact_fields'], ['unit_id' => $post_id]);
		if (isset($_POST['fstrb_contact_fields']) && is_array($_POST['fstrb_contact_fields'])) {
			foreach ($_POST['fstrb_contact_fields'] as $fid) {
				$wpdb->insert($t['unit_contact_fields'], [
					'unit_id' => $post_id,
					'field_id' => (int)$fid
				]);
			}
		}
	}
}

add_action('init', ['FSTRB_CPT', 'register']);
