<?php
if (!defined('ABSPATH')) exit;

class FSTRB_DB {
	public static function tables() {
		global $wpdb;
		$p = $wpdb->prefix;
		return [
			'services' => $p . 'fstrb_services',
			'unit_services' => $p . 'fstrb_unit_services',
			'rates' => $p . 'fstrb_rates',
			'bookings' => $p . 'fstrb_bookings',
			'booking_services' => $p . 'fstrb_booking_services',
			'booking_days' => $p . 'fstrb_booking_days',
			'ical_feeds' => $p . 'fstrb_ical_feeds',
			'service_meta' => $p . 'fstrb_service_meta',
			'contact_fields' => $p . 'fstrb_contact_fields',
			'contact_field_meta' => $p . 'fstrb_contact_field_meta',
			'unit_contact_fields' => $p . 'fstrb_unit_contact_fields',
			'booking_meta' => $p . 'fstrb_booking_meta',
		];
	}

	public static function activate() {
		self::perform_migration();
		self::migrate();
		self::schedule_cron();
	}

	public static function deactivate() {
		wp_clear_scheduled_hook('fstrb_ical_import_cron');
	}

	public static function schedule_cron() {
		if (!wp_next_scheduled('fstrb_ical_import_cron')) {
			wp_schedule_event(time() + 300, 'hourly', 'fstrb_ical_import_cron');
		}
	}

	public static function perform_migration() {
		global $wpdb;
		$p = $wpdb->prefix;

		// 1. Rename Tables
		$tables_to_rename = [
			'ac_services' => 'fstrb_services',
			'ac_unit_services' => 'fstrb_unit_services',
			'ac_rates' => 'fstrb_rates',
			'ac_bookings' => 'fstrb_bookings',
			'ac_booking_services' => 'fstrb_booking_services',
			'ac_booking_days' => 'fstrb_booking_days',
			'ac_ical_feeds' => 'fstrb_ical_feeds',
			'ac_service_meta' => 'fstrb_service_meta',
		];

		foreach ($tables_to_rename as $old => $new) {
			$old_full = $p . $old;
			$new_full = $p . $new;
			if ($wpdb->get_var("SHOW TABLES LIKE '$old_full'") && !$wpdb->get_var("SHOW TABLES LIKE '$new_full'")) {
				$wpdb->query("RENAME TABLE $old_full TO $new_full");
			}
		}

		// 2. Update Post Type
		$wpdb->query($wpdb->prepare(
			"UPDATE {$wpdb->posts} SET post_type = %s WHERE post_type = %s",
			'fstrb_unit', 'ac_unit'
		));

		// 3. Update Taxonomy
		$wpdb->query($wpdb->prepare(
			"UPDATE {$wpdb->term_taxonomy} SET taxonomy = %s WHERE taxonomy = %s",
			'fstrb_amenity', 'ac_amenity'
		));

		// 4. Update Meta Keys
		$wpdb->query("UPDATE {$wpdb->postmeta} SET meta_key = REPLACE(meta_key, '_acb_', '_fstrb_') WHERE meta_key LIKE '_acb_%'");

		// 5. Update Options
		$wpdb->query("UPDATE {$wpdb->options} SET option_name = REPLACE(option_name, 'acb_', 'fstrb_') WHERE option_name LIKE 'acb_%'");
	}

	public static function migrate() {
		global $wpdb;
		require_once ABSPATH . 'wp-admin/includes/upgrade.php';

		$charset = $wpdb->get_charset_collate();
		$t = self::tables();

		$sql = [];

		$sql[] = "CREATE TABLE {$t['services']} (
			id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
			name VARCHAR(190) NOT NULL,
			code VARCHAR(64) NOT NULL,
			pricing_type ENUM('per_stay','per_night','per_person','per_person_night','free') NOT NULL DEFAULT 'per_stay',
			price DECIMAL(10,2) NOT NULL DEFAULT 0,
			description TEXT NULL,
			is_active TINYINT(1) NOT NULL DEFAULT 1,
			is_optional TINYINT(1) NOT NULL DEFAULT 1,
			sort_order INT NOT NULL DEFAULT 0,
			created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
			updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
			PRIMARY KEY  (id),
			UNIQUE KEY code (code)
		) $charset;";

		$sql[] = "CREATE TABLE {$t['service_meta']} (
			id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
			service_id BIGINT UNSIGNED NOT NULL,
			meta_key VARCHAR(190) NOT NULL,
			meta_value LONGTEXT NULL,
			locale VARCHAR(10) NULL,
			PRIMARY KEY (id),
			KEY service_id (service_id)
		) $charset;";

		$sql[] = "CREATE TABLE {$t['contact_fields']} (
			id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
			field_key VARCHAR(64) NOT NULL,
			is_active TINYINT(1) NOT NULL DEFAULT 1,
			sort_order INT NOT NULL DEFAULT 0,
			PRIMARY KEY (id),
			UNIQUE KEY field_key (field_key)
		) $charset;";

		$sql[] = "CREATE TABLE {$t['unit_contact_fields']} (
			unit_id BIGINT UNSIGNED NOT NULL,
			field_id BIGINT UNSIGNED NOT NULL,
			PRIMARY KEY (unit_id, field_id),
			KEY field_id (field_id)
		) $charset;";

		$sql[] = "CREATE TABLE {$t['contact_field_meta']} (
			id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
			field_id BIGINT UNSIGNED NOT NULL,
			meta_key VARCHAR(190) NOT NULL,
			meta_value LONGTEXT NULL,
			locale VARCHAR(10) NULL,
			PRIMARY KEY (id),
			KEY field_id (field_id)
		) $charset;";

		$sql[] = "CREATE TABLE {$t['booking_meta']} (
			id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
			booking_id BIGINT UNSIGNED NOT NULL,
			meta_key VARCHAR(190) NOT NULL,
			meta_value LONGTEXT NULL,
			PRIMARY KEY (id),
			KEY booking_id (booking_id)
		) $charset;";

		$sql[] = "CREATE TABLE {$t['unit_services']} (
			unit_id BIGINT UNSIGNED NOT NULL,
			service_id BIGINT UNSIGNED NOT NULL,
			currency CHAR(3) NOT NULL DEFAULT 'EUR',
			is_optional TINYINT(1) NOT NULL DEFAULT 1,
			price_override DECIMAL(10,2) NULL,
			pricing_type ENUM('per_stay','per_night','per_person','per_person_night','free') NULL,
			PRIMARY KEY (unit_id, service_id, currency),
			KEY service_id (service_id)
		) $charset;";

		$sql[] = "CREATE TABLE {$t['rates']} (
			id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
			unit_id BIGINT UNSIGNED NOT NULL,
			name VARCHAR(190) NOT NULL,
			date_from DATE NOT NULL,
			date_to DATE NOT NULL,
			rule ENUM('per_person_night','per_unit_night','weekend','workday','christmas','new_year','winter','summer','off_season','fixed') NOT NULL DEFAULT 'per_unit_night',
			operation ENUM('replace','discount_percent','discount_amount') NOT NULL DEFAULT 'replace',
			price DECIMAL(10,2) NOT NULL DEFAULT 0,
			currency CHAR(3) NOT NULL DEFAULT 'EUR',
			min_nights INT NULL,
			priority INT NOT NULL DEFAULT 100,
			created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
			updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
			PRIMARY KEY  (id),
			KEY unit_id (unit_id),
			KEY date_range (date_from, date_to),
			KEY priority (priority)
		) $charset;";

		$sql[] = "CREATE TABLE {$t['bookings']} (
			id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
			unit_id BIGINT UNSIGNED NOT NULL,
			status ENUM('pending','confirmed','cancelled','blocked') NOT NULL DEFAULT 'pending',
			source VARCHAR(50) NOT NULL DEFAULT 'direct',
			date_from DATE NOT NULL,
			date_to DATE NOT NULL,
			guests INT NOT NULL DEFAULT 1,
			customer_name VARCHAR(190) NULL,
			customer_lastname VARCHAR(190) NULL,
			customer_gender VARCHAR(20) NULL,
			customer_email VARCHAR(190) NULL,
			customer_phone VARCHAR(64) NULL,
			note TEXT NULL,
			discount_type ENUM('none','percent','amount') NOT NULL DEFAULT 'none',
			discount_value DECIMAL(10,2) NOT NULL DEFAULT 0,
			total_price DECIMAL(10,2) NULL,
			currency CHAR(3) NOT NULL DEFAULT 'EUR',
			ical_uid VARCHAR(255) NULL,
			ical_summary VARCHAR(500) NULL,
			ical_description TEXT NULL,
			external_booking_id VARCHAR(100) NULL,
			created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
			updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
			PRIMARY KEY (id),
			KEY unit_id (unit_id),
			KEY status (status),
			KEY date_range (date_from, date_to),
			KEY external_booking_id (external_booking_id),
			KEY ical_uid (ical_uid)
		) $charset;";

		$sql[] = "CREATE TABLE {$t['booking_services']} (
			booking_id BIGINT UNSIGNED NOT NULL,
			service_id BIGINT UNSIGNED NOT NULL,
			qty DECIMAL(10,2) NOT NULL DEFAULT 1,
			price DECIMAL(10,2) NOT NULL DEFAULT 0,
			PRIMARY KEY (booking_id, service_id),
			KEY service_id (service_id)
		) $charset;";

		// "Lock" table: one row per day per unit
		$sql[] = "CREATE TABLE {$t['booking_days']} (
			unit_id BIGINT UNSIGNED NOT NULL,
			day DATE NOT NULL,
			booking_id BIGINT UNSIGNED NOT NULL,
			PRIMARY KEY (unit_id, day),
			KEY booking_id (booking_id)
		) $charset;";

		// iCal feeds per unit (imports)
		$sql[] = "CREATE TABLE {$t['ical_feeds']} (
			id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
			unit_id BIGINT UNSIGNED NOT NULL,
			provider VARCHAR(50) NOT NULL DEFAULT 'other',
			url TEXT NOT NULL,
			is_active TINYINT(1) NOT NULL DEFAULT 1,
			last_import_at DATETIME NULL,
			last_error TEXT NULL,
			created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
			updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
			PRIMARY KEY (id),
			KEY unit_id (unit_id),
			KEY is_active (is_active)
		) $charset;";

		foreach ($sql as $q) {
			dbDelta($q);
		}

		// Ensure provider/source columns are VARCHAR(50)
		$wpdb->query("ALTER TABLE {$t['ical_feeds']} MODIFY provider VARCHAR(50) NOT NULL DEFAULT 'other'");
		$wpdb->query("ALTER TABLE {$t['bookings']} MODIFY source VARCHAR(50) NOT NULL DEFAULT 'direct'");

		// Seed a couple of common services if empty
		$count = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$t['services']}");
		if ($count === 0) {
			$wpdb->insert($t['services'], ['name'=>'Cleaning', 'code'=>'CLEANING', 'pricing_type'=>'per_stay', 'price'=>0]);
			$wpdb->insert($t['services'], ['name'=>'Bed Linen', 'code'=>'LINEN', 'pricing_type'=>'per_stay', 'price'=>0]);
			$wpdb->insert($t['services'], ['name'=>'Parking', 'code'=>'PARKING', 'pricing_type'=>'per_stay', 'price'=>0]);
			$wpdb->insert($t['services'], ['name'=>'Extra Bed', 'code'=>'EXTRA_BED', 'pricing_type'=>'per_night', 'price'=>0]);
			$wpdb->insert($t['services'], ['name'=>'Tourist Tax', 'code'=>'TOURIST_TAX', 'pricing_type'=>'per_person_night', 'price'=>0]);
			$wpdb->insert($t['services'], ['name'=>'Air Conditioning', 'code'=>'AIR_CONDITIONING', 'pricing_type'=>'per_stay', 'price'=>0]);
			$wpdb->insert($t['services'], ['name'=>'WIFI', 'code'=>'WIFI', 'pricing_type'=>'per_stay', 'price'=>0]);
		}
	}

	public static function unit_exists($unit_id) {
		$p = get_post($unit_id);
		return $p && $p->post_type === 'fstrb_unit' && $p->post_status !== 'trash';
	}
}
