<?php
if (!defined('ABSPATH')) exit;

class FSTRB_iCal {
	public static function detect_provider($url) {
		$url = strtolower($url);
		// Airbnb - supports multiple country domains
		if (strpos($url, 'airbnb.') !== false) return 'airbnb';
		// Booking.com - supports multiple country domains
		if (strpos($url, 'booking.') !== false) return 'booking';
		if (strpos($url, 'e-chalupy.cz') !== false) return 'e_chalupy';
		if (strpos($url, 'megaubytovanie.sk') !== false) return 'megaubytovanie';
		if (strpos($url, 'hauzi.sk') !== false || strpos($url, 'hauzi.cz') !== false) return 'hauzi';
		if (strpos($url, 'bezrealitky.cz') !== false || strpos($url, 'bezrealitky.sk') !== false) return 'bezrealitky';
		
		// Extract domain name for unknown providers
		$parsed = parse_url($url);
		if (isset($parsed['host'])) {
			// Remove www. prefix if present
			$domain = preg_replace('/^www\./i', '', $parsed['host']);
			return $domain;
		}
		
		return 'other';
	}

	public static function export($req) {
		global $wpdb;
		$t = FSTRB_DB::tables();

		$unit_id = (int)$req->get_param('unit_id');
		$token = sanitize_text_field($req->get_param('token'));

		if (!FSTRB_DB::unit_exists($unit_id)) return new WP_REST_Response('unit_not_found', 404);

		$saved = get_post_meta($unit_id, '_fstrb_ical_token', true);
		if (!$saved || !hash_equals($saved, $token)) return new WP_REST_Response('forbidden', 403);

		$rows = $wpdb->get_results($wpdb->prepare(
			"SELECT id, date_from, date_to, status, source FROM {$t['bookings']}
			 WHERE unit_id=%d AND status IN ('pending','confirmed','blocked')",
			$unit_id
		), ARRAY_A);

		$lines = [];
		$lines[] = "BEGIN:VCALENDAR";
		$lines[] = "VERSION:2.0";
		$lines[] = "PRODID:-//ApartmentsBooking//EN";
		$lines[] = "CALSCALE:GREGORIAN";

		foreach ($rows as $r) {
			$uid = 'fstrb-' . $unit_id . '-' . $r['id'] . '@' . parse_url(home_url(), PHP_URL_HOST);
			$dtstart = str_replace('-', '', $r['date_from']);
			$dtend = str_replace('-', '', $r['date_to']); // iCal DTEND is exclusive for all-day
			$summary = ($r['status'] === 'blocked') ? 'Blocked' : 'Reserved';
			$lines[] = "BEGIN:VEVENT";
			$lines[] = "UID:$uid";
			$lines[] = "DTSTAMP:" . gmdate('Ymd\THis\Z');
			$lines[] = "DTSTART;VALUE=DATE:$dtstart";
			$lines[] = "DTEND;VALUE=DATE:$dtend";
			$lines[] = "SUMMARY:" . self::escape($summary);
			$lines[] = "END:VEVENT";
		}

		$lines[] = "END:VCALENDAR";
		$ical = implode("\r\n", $lines) . "\r\n";

		header('Content-Type: text/calendar; charset=UTF-8');
		header('Content-Disposition: inline; filename="unit-' . $unit_id . '.ics"');
		echo $ical;
		exit;
	}

	private static function escape($s) {
		$s = (string)$s;
		$s = str_replace("\\", "\\\\", $s);
		$s = str_replace("\n", "\\n", $s);
		$s = str_replace([",",";"], ["\\,","\\;"], $s);
		return $s;
	}

	public static function import_unit_feeds($unit_id) {
		global $wpdb;
		$t = FSTRB_DB::tables();

		$feeds = $wpdb->get_results($wpdb->prepare(
			"SELECT * FROM {$t['ical_feeds']} WHERE unit_id=%d AND is_active=1",
			$unit_id
		), ARRAY_A);

		if (!$feeds) return ['ok'=>true, 'message'=>'No active imports.'];

		$imported = 0;
		foreach ($feeds as $f) {
			$url = $f['url'];
			$provider = $f['provider'];
			$res = wp_remote_get($url, ['timeout'=>20, 'redirection'=>3]);
			if (is_wp_error($res)) {
				$wpdb->update($t['ical_feeds'], ['last_error'=>$res->get_error_message(), 'last_import_at'=>current_time('mysql')], ['id'=>$f['id']]);
				continue;
			}
			$body = wp_remote_retrieve_body($res);
			try {
				$events = self::parse_ics_events($body);
				$n = self::upsert_blocks_from_events($unit_id, $events, $provider);
				$imported += $n;
				$wpdb->update($t['ical_feeds'], ['last_error'=>null, 'last_import_at'=>current_time('mysql')], ['id'=>$f['id']]);
			} catch (Exception $e) {
				$wpdb->update($t['ical_feeds'], ['last_error'=>$e->getMessage(), 'last_import_at'=>current_time('mysql')], ['id'=>$f['id']]);
			}
		}

		return ['ok'=>true, 'message'=>"Done. Imported/updated blocks: $imported"];
	}

	public static function parse_ics_events($ics) {
		// Enhanced iCal parser: reads DTSTART/DTEND/UID/SUMMARY/DESCRIPTION per VEVENT
		$ics = str_replace("\r\n", "\n", $ics);
		$lines = explode("\n", $ics);

		// unfold lines (RFC: lines starting with space are continuation)
		$unfolded = [];
		foreach ($lines as $line) {
			if ($line === '') continue;
			if (preg_match('/^[ \t]/', $line) && !empty($unfolded)) {
				$unfolded[count($unfolded)-1] .= trim($line);
			} else {
				$unfolded[] = trim($line);
			}
		}

		$events = [];
		$in = false;
		$cur = [];
		foreach ($unfolded as $line) {
			if ($line === 'BEGIN:VEVENT') { $in = true; $cur = []; continue; }
			if ($line === 'END:VEVENT') {
				if (!empty($cur['DTSTART']) && !empty($cur['DTEND'])) {
					$events[] = [
						'start' => $cur['DTSTART'],
						'end' => $cur['DTEND'],
						'uid' => $cur['UID'] ?? null,
						'summary' => $cur['SUMMARY'] ?? null,
						'description' => $cur['DESCRIPTION'] ?? null,
					];
				}
				$in = false;
				$cur = [];
				continue;
			}
			if (!$in) continue;

			if (stripos($line, 'DTSTART') === 0) {
				$cur['DTSTART'] = self::parse_date_value($line);
			} elseif (stripos($line, 'DTEND') === 0) {
				$cur['DTEND'] = self::parse_date_value($line);
			} elseif (stripos($line, 'UID:') === 0) {
				$cur['UID'] = self::parse_text_value($line);
			} elseif (stripos($line, 'SUMMARY:') === 0) {
				$cur['SUMMARY'] = self::parse_text_value($line);
			} elseif (stripos($line, 'DESCRIPTION:') === 0) {
				$cur['DESCRIPTION'] = self::parse_text_value($line);
			}
		}

		return $events;
	}

	private static function parse_date_value($line) {
		// Examples:
		// DTSTART;VALUE=DATE:20260105
		// DTSTART:20260105
		// DTSTART:20260226T140000
		$parts = explode(':', $line, 2);
		$val = $parts[1] ?? '';
		$val = trim($val);

		if (preg_match('/^(\d{8})(T\d{6}Z?)?$/', $val, $m)) {
			$date = $m[1];
			return substr($date,0,4).'-'.substr($date,4,2).'-'.substr($date,6,2);
		}

		throw new Exception('Unsupported DTSTART/DTEND format.');
	}

	private static function parse_text_value($line) {
		// Extract text value after colon, unescape iCal special chars
		$parts = explode(':', $line, 2);
		$val = $parts[1] ?? '';
		$val = trim($val);
		// Unescape iCal encoding
		$val = str_replace(['\\n', '\\,', '\\;', '\\\\'], ["\n", ',', ';', '\\'], $val);
		return $val;
	}

	private static function extract_booking_id($summary, $description, $provider) {
		// Try to extract reservation/booking ID from summary or description
		$text = ($summary ?? '') . ' ' . ($description ?? '');
		
		// Airbnb patterns
		if ($provider === 'airbnb') {
			if (preg_match('/\b(HM[A-Z0-9]{10,})\b/i', $text, $m)) return $m[1];
			if (preg_match('/confirmation code[:\s]+([A-Z0-9]{8,})/i', $text, $m)) return $m[1];
			if (preg_match('/reservation[:\s]+([A-Z0-9]{8,})/i', $text, $m)) return $m[1];
		}
		
		// Booking.com patterns
		if ($provider === 'booking') {
			if (preg_match('/booking\.com[\s\w]*[:\s]+(\d{10,})/i', $text, $m)) return $m[1];
			if (preg_match('/reservation number[:\s]+(\d{8,})/i', $text, $m)) return $m[1];
			if (preg_match('/\b(\d{10})\b/', $text, $m)) return $m[1]; // 10-digit number
		}
		
		// Generic patterns
		if (preg_match('/\b([A-Z]{2}\d{8,})\b/', $text, $m)) return $m[1];
		if (preg_match('/booking[:\s#]+([A-Z0-9-]{6,})/i', $text, $m)) return $m[1];
		if (preg_match('/reservation[:\s#]+([A-Z0-9-]{6,})/i', $text, $m)) return $m[1];
		
		return null;
	}

	private static function upsert_blocks_from_events($unit_id, $events, $provider) {
		// Strategy (MVP):
		// - Create "blocked" bookings for each event if not exists with same date_from/date_to/source
		// - Do NOT delete old blocks automatically (to avoid accidentally freeing dates)
		// - Store iCal metadata (UID, summary, description, external booking ID)
		global $wpdb;
		$t = FSTRB_DB::tables();
		$count = 0;

		foreach ($events as $ev) {
			$df = $ev['start'];
			$dt = $ev['end'];
			if (strtotime($dt) <= strtotime($df)) continue;

			$source = 'ical_' . $provider;
			$ical_uid = $ev['uid'] ?? null;
			$ical_summary = $ev['summary'] ?? null;
			$ical_description = $ev['description'] ?? null;
			$external_booking_id = self::extract_booking_id($ical_summary, $ical_description, $provider);

			// Check if exists by UID first, then by date range
			$exists = null;
			if ($ical_uid) {
				$exists = (int)$wpdb->get_var($wpdb->prepare(
					"SELECT id FROM {$t['bookings']} WHERE unit_id=%d AND ical_uid=%s",
					$unit_id, $ical_uid
				));
			}
			if (!$exists) {
				$exists = (int)$wpdb->get_var($wpdb->prepare(
					"SELECT id FROM {$t['bookings']} WHERE unit_id=%d AND source=%s AND status='blocked' AND date_from=%s AND date_to=%s",
					$unit_id, $source, $df, $dt
				));
			}
			
			// If exists, update metadata
			if ($exists) {
				$wpdb->update($t['bookings'], [
					'ical_uid' => $ical_uid,
					'ical_summary' => $ical_summary,
					'ical_description' => $ical_description,
					'external_booking_id' => $external_booking_id,
				], ['id' => $exists]);
				continue;
			}

			// create with locking days (transaction per block)
			$wpdb->query('START TRANSACTION');
			$ok = $wpdb->insert($t['bookings'], [
				'unit_id'=>$unit_id,
				'status'=>'blocked',
				'source'=>$source,
				'date_from'=>$df,
				'date_to'=>$dt,
				'guests'=>1,
				'currency'=> (get_post_meta($unit_id, '_fstrb_currency', true) ?: 'EUR'),
				'ical_uid' => $ical_uid,
				'ical_summary' => $ical_summary,
				'ical_description' => $ical_description,
				'external_booking_id' => $external_booking_id,
			]);
			if (!$ok) { $wpdb->query('ROLLBACK'); continue; }
			$bid = (int)$wpdb->insert_id;

			$d1 = new DateTime($df);
			$d2 = new DateTime($dt);
			$conflict = false;
			while ($d1 < $d2) {
				$day = $d1->format('Y-m-d');
				$ins = $wpdb->query($wpdb->prepare(
					"INSERT INTO {$t['booking_days']} (unit_id, day, booking_id) VALUES (%d, %s, %d)",
					$unit_id, $day, $bid
				));
				if ($ins === false) { $conflict = true; break; }
				$d1->modify('+1 day');
			}
			if ($conflict) { $wpdb->query('ROLLBACK'); continue; }

			$wpdb->query('COMMIT');
			$count++;
		}

		return $count;
	}
}

// Cron hook
add_action('fstrb_ical_import_cron', function() {
	global $wpdb;
	$t = FSTRB_DB::tables();
	$unit_ids = $wpdb->get_col("SELECT DISTINCT unit_id FROM {$t['ical_feeds']} WHERE is_active=1");
	foreach ($unit_ids as $unit_id) {
		FSTRB_iCal::import_unit_feeds((int)$unit_id);
	}
});
