<?php
if (!defined('ABSPATH')) exit;

class FSTRB_REST {
	public static function register() {
		register_rest_route('fstrb/v1', '/calendar', [
			'methods' => 'GET',
			'callback' => [__CLASS__, 'calendar'],
			'permission_callback' => '__return_true',
			'args' => [
				'unit_id' => ['required'=>true],
				'month' => ['required'=>true], // YYYY-MM
				'months' => ['required'=>false, 'default'=>1],
			]
		]);

		register_rest_route('fstrb/v1', '/services', [
			'methods' => 'GET',
			'callback' => [__CLASS__, 'services'],
			'permission_callback' => '__return_true',
			'args' => [
				'unit_id' => ['required'=>true],
				'currency' => ['required'=>false, 'default'=>'EUR'],
			]
		]);

		register_rest_route('fstrb/v1', '/quote', [
			'methods' => 'POST',
			'callback' => [__CLASS__, 'quote'],
			'permission_callback' => '__return_true',
		]);

		register_rest_route('fstrb/v1', '/book', [
			'methods' => 'POST',
			'callback' => [__CLASS__, 'book'],
			// Public booking allowed, but still requires basic validation.
			'permission_callback' => '__return_true',
		]);

		register_rest_route('fstrb/v1', '/ical/export', [
			'methods' => 'GET',
			'callback' => ['FSTRB_iCal', 'export'],
			'permission_callback' => '__return_true',
		]);

		register_rest_route('fstrb/v1', '/ical/export.ics', [
			'methods' => 'GET',
			'callback' => ['FSTRB_iCal', 'export'],
			'permission_callback' => '__return_true',
		]);

		register_rest_route('fstrb/v1', '/ical/export/(?P<unit_id>\d+)/(?P<token>[a-zA-Z0-9]+)\.ics', [
			'methods' => 'GET',
			'callback' => ['FSTRB_iCal', 'export'],
			'permission_callback' => '__return_true',
		]);

		register_rest_route('fstrb/v1', '/contact-fields', [
			'methods' => 'GET',
			'callback' => [__CLASS__, 'contact_fields'],
			'permission_callback' => '__return_true',
			'args' => [
				'locale' => ['required'=>false],
			]
		]);
	}

	public static function calendar($req) {
		global $wpdb;
		$t = FSTRB_DB::tables();

		$unit_id = (int)$req->get_param('unit_id');
		$month = sanitize_text_field($req->get_param('month'));
		$months = max(1, min(12, (int)$req->get_param('months'))); // limit to 1-12 months

		if (!FSTRB_DB::unit_exists($unit_id)) {
			return new WP_REST_Response(['error'=>'unit_not_found'], 404);
		}
		if (get_post_meta($unit_id, '_fstrb_is_paused', true)) {
			return new WP_REST_Response(['error'=>'unit_paused'], 400);
		}

		if (!preg_match('/^\d{4}-\d{2}$/', $month)) {
			return new WP_REST_Response(['error'=>'invalid_month'], 400);
		}

		$from = $month . '-01';
		$dt = new DateTime($from);
		$dt_end = clone $dt;
		$dt_end->modify('+' . $months . ' months');

		$to = $dt_end->format('Y-m-d');

		// Get occupied days
		$rows = $wpdb->get_results($wpdb->prepare(
			"SELECT day, booking_id FROM {$t['booking_days']} WHERE unit_id=%d AND day >= %s AND day < %s",
			$unit_id, $from, $to
		), ARRAY_A);

		$occupied = [];
		foreach ($rows as $r) {
			$occupied[$r['day']] = (int)$r['booking_id'];
		}

		// For nicer UI: get booking ranges overlapping this month
		$ranges = $wpdb->get_results($wpdb->prepare(
			"SELECT id, date_from, date_to, status FROM {$t['bookings']}
			 WHERE unit_id=%d AND status IN ('pending','confirmed','cancelled','blocked')
			 AND NOT (date_to <= %s OR date_from >= %s)",
			$unit_id, $from, $to
		), ARRAY_A);

		return new WP_REST_Response([
			'month' => $month,
			'from' => $from,
			'to' => $to,
			'occupiedDays' => array_keys($occupied),
			'ranges' => $ranges,
		], 200);
	}

	public static function services($req) {
		global $wpdb;
		$t = FSTRB_DB::tables();
		$unit_id = (int)$req->get_param('unit_id');
		if (!FSTRB_DB::unit_exists($unit_id)) return new WP_REST_Response(['error'=>'unit_not_found'], 404);

		$currency = sanitize_text_field($req->get_param('currency') ?? 'EUR');
		
		// Fetch all services enabled for this unit (any currency)
		$sql = "SELECT s.id, s.name, s.code, 
					   s.pricing_type as global_pricing_type,
					   s.price as global_price,
					   s.description,
					   us.price_override,
					   us.pricing_type as override_pricing_type,
					   us.is_optional,
					   us.currency
				FROM {$t['services']} s
				INNER JOIN {$t['unit_services']} us ON us.service_id=s.id
				WHERE us.unit_id=%d AND s.is_active=1
				ORDER BY s.sort_order ASC, s.name ASC, (us.currency = %s) DESC, (us.currency = 'EUR') DESC";

		$req_locale = sanitize_text_field($req->get_param('locale') ?? '');
		
		$raw_items = $wpdb->get_results($wpdb->prepare($sql, $unit_id, $currency), ARRAY_A);
		$items = [];
		$seen = [];

		foreach ($raw_items as $row) {
			$sid = $row['id'];
			if (isset($seen[$sid])) continue;
			$seen[$sid] = true;

			$item = [
				'id' => $row['id'],
				'name' => __($row['name'], 'free-short-term-rental-booking'),
				'code' => $row['code'],
				'description' => $row['description'],
				'pricing_type' => $row['override_pricing_type'] ?: $row['global_pricing_type'],
				'is_optional' => $row['is_optional'],
			];

			// Determine price: unit_override (for this curr) > meta(price_CUR) > global_price
			// Since our query orders by matching currency first, we can check if this row IS the matching currency
			if ($row['currency'] === $currency && !is_null($row['price_override'])) {
				$item['price'] = (float)$row['price_override'];
			} else {
				// Fallback to meta or global
				$meta_price = $wpdb->get_var($wpdb->prepare("SELECT meta_value FROM {$t['service_meta']} WHERE service_id=%d AND meta_key=%s", $row['id'], 'price_' . $currency));
				$item['price'] = !is_null($meta_price) ? (float)$meta_price : (float)$row['global_price'];
			}

			if ($req_locale) {
				$meta_sql = $wpdb->prepare("SELECT meta_key, meta_value, locale FROM {$t['service_meta']} WHERE service_id=%d AND (locale IS NULL OR locale = %s OR locale = '')", $item['id'], $req_locale);
			} else {
				$meta_sql = $wpdb->prepare("SELECT meta_key, meta_value, locale FROM {$t['service_meta']} WHERE service_id=%d", $item['id']);
			}
			$meta_rows = $wpdb->get_results($meta_sql, ARRAY_A);
			$meta = [];
			foreach ($meta_rows as $mr) {
				if (strpos($mr['meta_key'], 'price_') === 0) continue;
				$meta[] = [
					'key' => $mr['meta_key'],
					'value' => $mr['meta_value'],
					'locale' => $mr['locale']
				];
			}
			$item['meta_data'] = $meta;
			$items[] = $item;
		}

		return new WP_REST_Response(['services' => $items], 200);
	}

	public static function contact_fields($req) {
		global $wpdb;
		$t = FSTRB_DB::tables();
		$locale = sanitize_text_field($req->get_param('locale') ?? '');
		$unit_id = (int)$req->get_param('unit_id');

		if ($unit_id > 0) {
			$sql = $wpdb->prepare("SELECT f.id, f.field_key 
					FROM {$t['contact_fields']} f
					INNER JOIN {$t['unit_contact_fields']} uf ON uf.field_id = f.id
					WHERE f.is_active=1 AND uf.unit_id=%d 
					ORDER BY f.sort_order ASC", $unit_id);
		} else {
			$sql = "SELECT id, field_key FROM {$t['contact_fields']} WHERE is_active=1 ORDER BY sort_order ASC";
		}
		
		$fields = $wpdb->get_results($sql, ARRAY_A);
		
		$res = [];
		foreach ($fields as $f) {
			// Get label for current locale then fallback to global (NULL)
			$label = '';
			if ($locale) {
				$label = $wpdb->get_var($wpdb->prepare("SELECT meta_value FROM {$t['contact_field_meta']} WHERE field_id=%d AND meta_key='label' AND locale=%s", $f['id'], $locale));
			}
			if (!$label) {
				$label = $wpdb->get_var($wpdb->prepare("SELECT meta_value FROM {$t['contact_field_meta']} WHERE field_id=%d AND meta_key='label' AND (locale IS NULL OR locale = '')", $f['id']));
			}
			if (!$label) $label = $f['field_key'];

			$res[] = [
				'key' => $f['field_key'],
				'label' => $label
			];
		}

		return new WP_REST_Response(['fields' => $res], 200);
	}

	private static function date_range_days($date_from, $date_to) {
		$days = [];
		$d1 = new DateTime($date_from);
		$d2 = new DateTime($date_to);
		// nights: [from, to)
		while ($d1 < $d2) {
			$days[] = $d1->format('Y-m-d');
			$d1->modify('+1 day');
		}
		return $days;
	}

	private static function base_price($unit_id, $date_from, $date_to, $guests, $currency = 'EUR') {
		// MVP pricing:
		// - base mode from post meta
		// - optional seasonal override via fstrb_rates (highest priority lowest number)
		$mode = get_post_meta($unit_id, '_fstrb_price_mode', true) ?: 'per_unit_night';
		
		$get_meta = function($key) use ($unit_id, $currency) {
			$val = get_post_meta($unit_id, $key . '_' . $currency, true);
			if ($val === '' && $currency !== 'EUR') $val = get_post_meta($unit_id, $key . '_EUR', true);
			if ($val === '') $val = get_post_meta($unit_id, $key, true);
			return (float)$val;
		};

		$pppn = $get_meta('_fstrb_price_per_person_night');
		$pun  = $get_meta('_fstrb_price_per_unit_night');
		$per_stay = $get_meta('_fstrb_price_per_stay');
		$per_night = $get_meta('_fstrb_price_per_night');
		$per_person = $get_meta('_fstrb_price_per_person');

		$nights = max(0, (new DateTime($date_to))->diff(new DateTime($date_from))->days);
		if ($nights <= 0) return [0, 0, 0];

		$base = 0;
		switch ($mode) {
			case 'per_person_night':
				$base = $pppn * $guests * $nights;
				break;
			case 'per_unit_night':
				$base = $pun * $nights;
				break;
			case 'per_stay':
				$base = $per_stay;
				break;
			case 'per_night':
				$base = $per_night * $nights;
				break;
			case 'per_person':
				$base = $per_person * $guests;
				break;
			default:
				$base = $pun * $nights; // fallback to per_unit_night
		}

		// Rates override: sum per-night based on matching rates (simple: pick best rate per day)
		global $wpdb;
		$t = FSTRB_DB::tables();

		$days = self::date_range_days($date_from, $date_to);
		if (!$days) return [$base, $nights, 0];

		// Fetch all rates overlapping
		$rates = $wpdb->get_results($wpdb->prepare(
			"SELECT * FROM {$t['rates']}
			 WHERE unit_id=%d AND currency=%s AND NOT (date_to < %s OR date_from > %s)
			 ORDER BY priority ASC, id ASC",
			$unit_id, $currency, $date_from, $date_to
		), ARRAY_A);

		if (!$rates) return [$base, $nights, 0];

		$sum = 0.0;
		$total_discount = 0.0; // Track total discount amount

		foreach ($days as $day) {
			$applied = false;
			foreach ($rates as $r) {
				if ($day < $r['date_from'] || $day > $r['date_to']) continue;
				
				$r_price = (float)$r['price'];
				$r_op = $r['operation'] ?? 'replace'; // Default for older DB entries
				
				// Calculate "Base Price" for this specific day/guest config (for discount calculation)
				$day_base = 0;
				switch ($mode) {
					case 'per_person_night': $day_base = $pppn * $guests; break;
					case 'per_unit_night':   $day_base = $pun; break;
					case 'per_night':        $day_base = $per_night; break;
					default: $day_base = $pun;
				}

				$dt_obj = new DateTime($day);
				$dow = (int)$dt_obj->format('N');
				$month = (int)$dt_obj->format('n');
				$md = $dt_obj->format('m-d');

				$matches_rule = false;
				switch ($r['rule']) {
					case 'per_person_night': 
					case 'per_unit_night':
					case 'fixed':
						$matches_rule = true; 
						break;
					case 'weekend':
						if ($dow >= 5) $matches_rule = true;
						break;
					case 'workday':
						if ($dow < 5) $matches_rule = true;
						break;
					case 'christmas':
						if (in_array($md, ['12-24','12-25','12-26'])) $matches_rule = true;
						break;
					case 'new_year':
						if ($md === '12-31') $matches_rule = true;
						break;
					case 'winter':
						if (in_array($month, [12,1,2,3])) $matches_rule = true;
						break;
					case 'summer':
						if (in_array($month, [7,8])) $matches_rule = true;
						break;
					case 'off_season':
						if (in_array($month, [4,5,6,9,10,11])) $matches_rule = true;
						break;
				}

				if (!$matches_rule) continue;

				if ($r_op === 'replace') {
					// behaviors based on RULE (legacy)
					switch ($r['rule']) {
						case 'per_person_night': $sum += $r_price * $guests; $applied = true; break;
						case 'per_unit_night': 
						case 'weekend':
						case 'workday':
						case 'christmas':
						case 'new_year':
						case 'winter':
						case 'summer':
						case 'off_season':
							$sum += $r_price; $applied = true; 
							break;
						case 'fixed':
							$sum += $r_price; $applied = true;
							break;
					}
				} elseif ($r_op === 'discount_percent') {
					 // Discount % from the Base Price of that day
					 $discount_amount = $day_base * ($r_price / 100);
					 $final = max(0, $day_base - $discount_amount);
					 $sum += $final;
					 $total_discount += $discount_amount;
					 $applied = true;
					 
				} elseif ($r_op === 'discount_amount') {
					 // Discount fixed amount from the Base Price
					 $final = max(0, $day_base - $r_price);
					 $sum += $final;
					 $total_discount += ($day_base - $final);
					 $applied = true;
				}

				if ($applied) break;
			}
			if (!$applied) {
				// fallback to base per-night calculation
				switch ($mode) {
					case 'per_person_night':
						$sum += ($pppn * $guests);
						break;
					case 'per_unit_night':
						$sum += $pun;
						break;
					case 'per_night':
						$sum += $per_night;
						break;
					case 'per_stay':
					case 'per_person':
						// For per_stay and per_person, we don't add per-day, already counted in base
						break;
					default:
						$sum += $pun;
				}
			}
		}

		// For per_stay and per_person modes, if rates were applied... 
		// Logic is tricky because these modes aren't per-night.
		// But our discount logic allows per-night discounts. 
		// If a discount was applied to the "Base Daily Price", we implicitly converted the booking to daily calculation.
		// Ideally, if any rate is applied, we trust $sum.
		
		return [$sum, $nights, $total_discount];
	}

	private static function services_price($unit_id, $services_selected, $date_from, $date_to, $guests, $currency = 'EUR') {
		global $wpdb;
		$t = FSTRB_DB::tables();
		if (!$services_selected) return [0, []];

		$service_ids = array_values(array_filter(array_map('intval', $services_selected)));
		if (!$service_ids) return [0, []];

		$placeholders = implode(',', array_fill(0, count($service_ids), '%d'));

		$rows = $wpdb->get_results($wpdb->prepare(
			"SELECT s.id, s.name, s.price as global_price,
					COALESCE(us.pricing_type, s.pricing_type) AS pricing_type,
					us.price_override,
					us.currency
			 FROM {$t['services']} s
			 INNER JOIN {$t['unit_services']} us ON us.service_id=s.id
			 WHERE us.unit_id=%d AND s.id IN ($placeholders) AND s.is_active=1
			 ORDER BY s.sort_order ASC, s.name ASC, (us.currency = %s) DESC, (us.currency = 'EUR') DESC",
			array_merge([$unit_id], $service_ids, [$currency])
		), ARRAY_A);

		$nights = max(0, (new DateTime($date_to))->diff(new DateTime($date_from))->days);

		$sum = 0.0;
		$line = [];
		$seen = [];
		foreach ($rows as $r) {
			if (isset($seen[$r['id']])) continue;
			$seen[$r['id']] = true;

			if ($r['currency'] === $currency && !is_null($r['price_override'])) {
				$p = (float)$r['price_override'];
			} else {
				$meta_price = $wpdb->get_var($wpdb->prepare("SELECT meta_value FROM {$t['service_meta']} WHERE service_id=%d AND meta_key=%s", $r['id'], 'price_' . $currency));
				$p = !is_null($meta_price) ? (float)$meta_price : (float)$r['global_price'];
			}
			$qty = 1.0;
			switch ($r['pricing_type']) {
				case 'free': $p = 0; break;
				case 'per_stay': $qty = 1; break;
				case 'per_night': $qty = $nights; break;
				case 'per_person': $qty = $guests; break;
				case 'per_person_night': $qty = $guests * $nights; break;
			}
			$item_total = $p * $qty;
			$sum += $item_total;
			$line[] = [
				'id' => (int)$r['id'],
				'name' => __($r['name'], 'free-short-term-rental-booking'),
				'unit_price' => $p,
				'qty' => $qty,
				'total' => $item_total
			];
		}
		return [$sum, $line];
	}

	public static function quote($req) {
		$data = $req->get_json_params();
		$unit_id = (int)($data['unit_id'] ?? 0);
		$date_from = sanitize_text_field($data['date_from'] ?? '');
		$date_to = sanitize_text_field($data['date_to'] ?? '');
		$guests = max(1, (int)($data['guests'] ?? 1));
		$services = $data['services'] ?? [];
		$currency = sanitize_text_field($data['currency'] ?? 'EUR');

		if (!FSTRB_DB::unit_exists($unit_id)) return new WP_REST_Response(['error'=>'unit_not_found'], 404);
		if (get_post_meta($unit_id, '_fstrb_is_paused', true)) return new WP_REST_Response(['error'=>'unit_paused'], 400);
		if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $date_from) || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $date_to)) {
			return new WP_REST_Response(['error'=>'invalid_dates'], 400);
		}
		if (strtotime($date_to) <= strtotime($date_from)) return new WP_REST_Response(['error'=>'invalid_range'], 400);

		[$base, $nights, $discount_total] = self::base_price($unit_id, $date_from, $date_to, $guests, $currency);

		// Validation: Min/Max stay
		$min_stay = (int)get_post_meta($unit_id, '_fstrb_min_stay', true);
		$max_stay = (int)get_post_meta($unit_id, '_fstrb_max_stay', true);
		if ($min_stay > 0 && $nights < $min_stay) return new WP_REST_Response(['error'=>'min_stay_violation', 'limit'=>$min_stay], 400);
		if ($max_stay > 0 && $nights > $max_stay) return new WP_REST_Response(['error'=>'max_stay_violation', 'limit'=>$max_stay], 400);

		[$svc_sum, $svc_lines] = self::services_price($unit_id, $services, $date_from, $date_to, $guests, $currency);

		$total = $base + $svc_sum;

		return new WP_REST_Response([
			'nights' => $nights,
			'base' => round($base, 2),
			'discount_total' => round($discount_total, 2),
			'services' => $svc_lines,
			'services_total' => round($svc_sum, 2),
			'total' => round($total, 2),
			'currency' => $currency,
		], 200);
	}

	public static function book($req) {
		global $wpdb;
		$t = FSTRB_DB::tables();

		$data = $req->get_json_params();
		$unit_id = (int)($data['unit_id'] ?? 0);
		$date_from = sanitize_text_field($data['date_from'] ?? '');
		$date_to = sanitize_text_field($data['date_to'] ?? '');
		$guests = max(1, (int)($data['guests'] ?? 1));
		$services = $data['services'] ?? [];
		$currency = sanitize_text_field($data['currency'] ?? 'EUR');
		$name = sanitize_text_field($data['name'] ?? '');
		$lastname = sanitize_text_field($data['lastname'] ?? '');
		$gender = sanitize_text_field($data['gender'] ?? '');
		$email = sanitize_email($data['email'] ?? '');
		$phone = sanitize_text_field($data['phone'] ?? '');
		$note = sanitize_textarea_field($data['note'] ?? '');
		$hp = $data['hp'] ?? '';
		$captcha = trim($data['captcha'] ?? '');
		$custom_fields = $data['custom_fields'] ?? [];

		if (!FSTRB_DB::unit_exists($unit_id)) {
			return new WP_REST_Response(['error'=>'unit_not_found'], 404);
		}

		// Anti-bot check
		if (!empty($hp)) {
			return new WP_REST_Response(['error'=>'robot_detected'], 400);
		}
		if ($captcha !== '5') {
			return new WP_REST_Response(['error'=>'captcha_invalid'], 400);
		}
		if (get_post_meta($unit_id, '_fstrb_is_paused', true)) {
			return new WP_REST_Response(['error'=>'unit_paused'], 400);
		}
		if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $date_from) || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $date_to)) {
			return new WP_REST_Response(['error'=>'invalid_dates'], 400);
		}
		if (strtotime($date_to) <= strtotime($date_from)) {
			return new WP_REST_Response(['error'=>'invalid_range'], 400);
		}
		if (!$name || !$lastname || !$email || !$phone) {
			return new WP_REST_Response(['error'=>'missing_contact'], 400);
		}

		// Quote
		[$base, $nights] = self::base_price($unit_id, $date_from, $date_to, $guests, $currency);

		// Validation: Min/Max stay
		$min_stay = (int)get_post_meta($unit_id, '_fstrb_min_stay', true);
		$max_stay = (int)get_post_meta($unit_id, '_fstrb_max_stay', true);
		if ($min_stay > 0 && $nights < $min_stay) {
			return new WP_REST_Response(['error'=>'min_stay_violation', 'limit'=>$min_stay], 400);
		}
		if ($max_stay > 0 && $nights > $max_stay) {
			return new WP_REST_Response(['error'=>'max_stay_violation', 'limit'=>$max_stay], 400);
		}

		[$svc_sum, $svc_lines] = self::services_price($unit_id, $services, $date_from, $date_to, $guests, $currency);
		$total = round($base + $svc_sum, 2);

		$status = 'pending';
		// Admin: Allow immediate confirmation
		if (!empty($data['confirm_now']) && current_user_can('manage_options')) {
			$status = 'confirmed';
		}

		// Transaction + lock days via booking_days UNIQUE(unit_id, day)
		$wpdb->query('START TRANSACTION');

		$ok = $wpdb->insert($t['bookings'], [
			'unit_id' => $unit_id,
			'status' => $status,
			'source' => 'direct',
			'date_from' => $date_from,
			'date_to' => $date_to,
			'guests' => $guests,
			'customer_name' => $name,
			'customer_lastname' => $lastname,
			'customer_gender' => $gender,
			'customer_email' => $email,
			'customer_phone' => $phone,
			'note' => $note,
			'total_price' => $total,
			'currency' => $currency,
		]);

		if (!$ok) {
			$wpdb->query('ROLLBACK');
			return new WP_REST_Response(['error'=>'db_insert_failed'], 500);
		}

		$booking_id = (int)$wpdb->insert_id;

		// Insert booking_days locks
		$days = self::date_range_days($date_from, $date_to);
		foreach ($days as $day) {
			$ins = $wpdb->query($wpdb->prepare(
				"INSERT INTO {$t['booking_days']} (unit_id, day, booking_id) VALUES (%d, %s, %d)",
				$unit_id, $day, $booking_id
			));
			if ($ins === false) {
				$wpdb->query('ROLLBACK');
				return new WP_REST_Response(['error'=>'date_conflict'], 409);
			}
		}

		// Booking services rows
		foreach ($svc_lines as $sl) {
			$wpdb->insert($t['booking_services'], [
				'booking_id' => $booking_id,
				'service_id' => (int)$sl['id'],
				'qty' => (float)$sl['qty'],
				'price' => (float)$sl['unit_price'],
			]);
		}

		// Save Custom Fields (Metadata)
		if (!empty($custom_fields)) {
			// Get allowed fields for this unit
			$allowed_keys = $wpdb->get_col($wpdb->prepare("
				SELECT f.field_key 
				FROM {$t['contact_fields']} f
				INNER JOIN {$t['unit_contact_fields']} uf ON uf.field_id = f.id
				WHERE uf.unit_id = %d AND f.is_active = 1
			", $unit_id));

			foreach ($custom_fields as $key => $val) {
				// Verify if this key is allowed
				if (in_array($key, $allowed_keys)) {
					$wpdb->insert($t['booking_meta'], [
						'booking_id' => $booking_id,
						'meta_key' => sanitize_text_field($key),
						'meta_value' => sanitize_text_field($val)
					]);
				}
			}
		}

		$wpdb->query('COMMIT');

		// Email Notification
		FSTRB_Emails::notify_admin_new_booking($booking_id);

		return new WP_REST_Response([
			'ok' => true,
			'booking_id' => $booking_id,
			'status' => $status,
			'total' => $total,
			'currency' => $currency,
		], 201);
	}
}

add_action('rest_api_init', ['FSTRB_REST', 'register']);
